'use strict';

Object.defineProperty(exports, "__esModule", {
  value: true
});

var _extends = Object.assign || function (target) { for (var i = 1; i < arguments.length; i++) { var source = arguments[i]; for (var key in source) { if (Object.prototype.hasOwnProperty.call(source, key)) { target[key] = source[key]; } } } return target; };

var _path = require('path');

var _path2 = _interopRequireDefault(_path);

var _eventKit = require('event-kit');

var _fsExtra = require('fs-extra');

var _fsExtra2 = _interopRequireDefault(_fsExtra);

var _actionPipeline = require('../action-pipeline');

var _compositeGitStrategy = require('../composite-git-strategy');

var _compositeGitStrategy2 = _interopRequireDefault(_compositeGitStrategy);

var _remote = require('./remote');

var _remote2 = _interopRequireDefault(_remote);

var _author = require('./author');

var _author2 = _interopRequireDefault(_author);

var _branch = require('./branch');

var _branch2 = _interopRequireDefault(_branch);

var _repositoryStates = require('./repository-states');

function _interopRequireDefault(obj) { return obj && obj.__esModule ? obj : { default: obj }; }

const MERGE_MARKER_REGEX = /^(>|<){7} \S+$/m;

// Internal option keys used to designate the desired initial state of a Repository.
const initialStateSym = Symbol('initialState');

class Repository {
  constructor(workingDirectoryPath, gitStrategy = null, options = {}) {
    this.workingDirectoryPath = workingDirectoryPath;
    this.git = gitStrategy || _compositeGitStrategy2.default.create(workingDirectoryPath);

    this.emitter = new _eventKit.Emitter();

    this.loadPromise = new Promise(resolve => {
      const sub = this.onDidChangeState(() => {
        if (!this.isLoading()) {
          resolve();
          sub.dispose();
        } else if (this.isDestroyed()) {
          sub.dispose();
        }
      });
    });

    this.pipelineManager = options.pipelineManager || (0, _actionPipeline.getNullActionPipelineManager)();
    this.transitionTo(options[initialStateSym] || _repositoryStates.Loading);
  }

  static absent(options) {
    return new Repository(null, null, _extends({ [initialStateSym]: _repositoryStates.Absent }, options));
  }

  static loadingGuess(options) {
    return new Repository(null, null, _extends({ [initialStateSym]: _repositoryStates.LoadingGuess }, options));
  }

  static absentGuess(options) {
    return new Repository(null, null, _extends({ [initialStateSym]: _repositoryStates.AbsentGuess }, options));
  }

  // State management //////////////////////////////////////////////////////////////////////////////////////////////////

  transition(currentState, StateConstructor, ...payload) {
    if (currentState !== this.state) {
      // Attempted transition from a non-active state, most likely from an asynchronous start() method.
      return Promise.resolve();
    }

    const nextState = new StateConstructor(this, ...payload);
    this.state = nextState;

    this.emitter.emit('did-change-state', { from: currentState, to: this.state });
    if (!this.isDestroyed()) {
      this.emitter.emit('did-update');
    }

    return this.state.start();
  }

  transitionTo(StateConstructor, ...payload) {
    return this.transition(this.state, StateConstructor, ...payload);
  }

  getLoadPromise() {
    return this.isAbsent() ? Promise.reject(new Error('An absent repository will never load')) : this.loadPromise;
  }

  /*
   * Use `callback` to request user input from all git strategies.
   */
  setPromptCallback(callback) {
    this.git.getImplementers().forEach(strategy => strategy.setPromptCallback(callback));
  }

  // Pipeline
  getPipeline(actionName) {
    const actionKey = this.pipelineManager.actionKeys[actionName];
    return this.pipelineManager.getPipeline(actionKey);
  }

  executePipelineAction(actionName, fn, ...args) {
    const pipeline = this.getPipeline(actionName);
    return pipeline.run(fn, this, ...args);
  }

  // Event subscription ////////////////////////////////////////////////////////////////////////////////////////////////

  onDidDestroy(callback) {
    return this.emitter.on('did-destroy', callback);
  }

  onDidChangeState(callback) {
    return this.emitter.on('did-change-state', callback);
  }

  onDidUpdate(callback) {
    return this.emitter.on('did-update', callback);
  }

  onMergeError(callback) {
    return this.emitter.on('merge-error', callback);
  }

  didMergeError() {
    return this.emitter.emit('merge-error');
  }

  // State-independent actions /////////////////////////////////////////////////////////////////////////////////////////
  // Actions that use direct filesystem access or otherwise don't need `this.git` to be available.

  async pathHasMergeMarkers(relativePath) {
    try {
      const contents = await _fsExtra2.default.readFile(_path2.default.join(this.getWorkingDirectoryPath(), relativePath), { encoding: 'utf8' });
      return MERGE_MARKER_REGEX.test(contents);
    } catch (e) {
      // EISDIR implies this is a submodule
      if (e.code === 'ENOENT' || e.code === 'EISDIR') {
        return false;
      } else {
        throw e;
      }
    }
  }

  async getMergeMessage() {
    try {
      const contents = await _fsExtra2.default.readFile(_path2.default.join(this.getGitDirectoryPath(), 'MERGE_MSG'), { encoding: 'utf8' });
      return contents.split(/\n/).filter(line => line.length > 0 && !line.startsWith('#')).join('\n');
    } catch (e) {
      return null;
    }
  }

  // State-independent accessors ///////////////////////////////////////////////////////////////////////////////////////

  getWorkingDirectoryPath() {
    return this.workingDirectoryPath;
  }

  setGitDirectoryPath(gitDirectoryPath) {
    this._gitDirectoryPath = gitDirectoryPath;
  }

  getGitDirectoryPath() {
    if (this._gitDirectoryPath) {
      return this._gitDirectoryPath;
    } else if (this.getWorkingDirectoryPath()) {
      return _path2.default.join(this.getWorkingDirectoryPath(), '.git');
    } else {
      // Absent/Loading/etc.
      return null;
    }
  }

  isInState(stateName) {
    return this.state.constructor.name === stateName;
  }

  toString() {
    return `Repository(state=${this.state.constructor.name}, workdir="${this.getWorkingDirectoryPath()}")`;
  }

  // Compound Getters //////////////////////////////////////////////////////////////////////////////////////////////////
  // Accessor methods for data derived from other, state-provided getters.

  async getCurrentBranch() {
    const branches = await this.getBranches();
    const head = branches.getHeadBranch();
    if (head.isPresent()) {
      return head;
    }

    const description = await this.getHeadDescription();
    return _branch2.default.createDetached(description);
  }

  async getUnstagedChanges() {
    const { unstagedFiles } = await this.getStatusBundle();
    return Object.keys(unstagedFiles).sort().map(filePath => {
      return { filePath, status: unstagedFiles[filePath] };
    });
  }

  async getStagedChanges() {
    const { stagedFiles } = await this.getStatusBundle();
    return Object.keys(stagedFiles).sort().map(filePath => {
      return { filePath, status: stagedFiles[filePath] };
    });
  }

  async getMergeConflicts() {
    const { mergeConflictFiles } = await this.getStatusBundle();
    return Object.keys(mergeConflictFiles).map(filePath => {
      return { filePath, status: mergeConflictFiles[filePath] };
    });
  }

  async isPartiallyStaged(fileName) {
    const { unstagedFiles, stagedFiles } = await this.getStatusBundle();
    const u = unstagedFiles[fileName];
    const s = stagedFiles[fileName];
    return u === 'modified' && s === 'modified' || u === 'modified' && s === 'added' || u === 'added' && s === 'deleted' || u === 'deleted' && s === 'modified';
  }

  async getRemoteForBranch(branchName) {
    const name = await this.getConfig(`branch.${branchName}.remote`);
    if (name === null) {
      return _remote.nullRemote;
    } else {
      return new _remote2.default(name);
    }
  }

  async saveDiscardHistory() {
    if (this.isDestroyed()) {
      return;
    }

    const historySha = await this.createDiscardHistoryBlob();
    if (this.isDestroyed()) {
      return;
    }
    await this.setConfig('atomGithub.historySha', historySha);
  }

  async getCommitter(options = {}) {
    const output = await this.getConfig(['--get-regexp', '^user.*'], options);
    const committer = { name: null, email: null };
    // todo (tt, 4/2018): do we need null byte terminated output here for Windows?
    if (output) {
      output.trim().split('\n').forEach(line => {
        if (line.includes('user.email')) {
          committer.email = line.slice(11);
        } else if (line.includes('user.name')) {
          committer.name = line.slice(10);
        }
      });
    }

    return committer.name !== null && committer.email !== null ? new _author2.default(committer.email, committer.name) : _author.nullAuthor;
  }

  async hasGitHubRemote(host, owner, name) {
    const remotes = await this.getRemotes();
    return remotes.matchingGitHubRepository(owner, name).length > 0;
  }
}

exports.default = Repository; // The methods named here will be delegated to the current State.
//
// Duplicated here rather than just using `expectedDelegates` directly so that this file is grep-friendly for answering
// the question of "what all can a Repository do exactly".

const delegates = ['isLoadingGuess', 'isAbsentGuess', 'isAbsent', 'isLoading', 'isEmpty', 'isPresent', 'isTooLarge', 'isDestroyed', 'isUndetermined', 'showGitTabInit', 'showGitTabInitInProgress', 'showGitTabLoading', 'showStatusBarTiles', 'hasDirectory', 'init', 'clone', 'destroy', 'refresh', 'observeFilesystemChange', 'updateCommitMessageAfterFileSystemChange', 'stageFiles', 'unstageFiles', 'stageFilesFromParentCommit', 'stageFileModeChange', 'stageFileSymlinkChange', 'applyPatchToIndex', 'applyPatchToWorkdir', 'commit', 'merge', 'abortMerge', 'checkoutSide', 'mergeFile', 'writeMergeConflictToIndex', 'checkout', 'checkoutPathsAtRevision', 'undoLastCommit', 'fetch', 'pull', 'push', 'setConfig', 'createBlob', 'expandBlobToFile', 'createDiscardHistoryBlob', 'updateDiscardHistory', 'storeBeforeAndAfterBlobs', 'restoreLastDiscardInTempFiles', 'popDiscardHistory', 'clearDiscardHistory', 'discardWorkDirChangesForPaths', 'getStatusBundle', 'getStatusesForChangedFiles', 'getFilePatchForPath', 'getStagedChangesPatch', 'readFileFromIndex', 'getLastCommit', 'getRecentCommits', 'getAuthors', 'getBranches', 'getHeadDescription', 'isMerging', 'isRebasing', 'getRemotes', 'addRemote', 'getAheadCount', 'getBehindCount', 'getConfig', 'unsetConfig', 'getBlobContents', 'hasDiscardHistory', 'getDiscardHistory', 'getLastHistorySnapshots', 'getOperationStates', 'setCommitMessage', 'getCommitMessage', 'fetchCommitMessageTemplate', 'getCache'];

for (let i = 0; i < delegates.length; i++) {
  const delegate = delegates[i];

  Repository.prototype[delegate] = function (...args) {
    return this.state[delegate](...args);
  };
}
//# sourceMappingURL=data:application/json;charset=utf-8;base64,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