'use strict';

Object.defineProperty(exports, "__esModule", {
  value: true
});

var _commit = require('../commit');

var _branchSet = require('../branch-set');

var _branchSet2 = _interopRequireDefault(_branchSet);

var _remoteSet = require('../remote-set');

var _remoteSet2 = _interopRequireDefault(_remoteSet);

var _operationStates = require('../operation-states');

var _multiFilePatch = require('../patch/multi-file-patch');

var _multiFilePatch2 = _interopRequireDefault(_multiFilePatch);

function _interopRequireDefault(obj) { return obj && obj.__esModule ? obj : { default: obj }; }

/**
 * Map of registered subclasses to allow states to transition to one another without circular dependencies.
 * Subclasses of State should call `State.register` to add themselves here.
 */
const stateConstructors = new Map();

/**
 * Base class for Repository states. Implements default "null" behavior.
 */
class State {
  constructor(repository) {
    this.repository = repository;
  }

  static register(Subclass) {
    stateConstructors.set(Subclass.name, Subclass);
  }

  // This state has just been entered. Perform any asynchronous initialization that needs to occur.
  start() {
    return Promise.resolve();
  }

  // State probe predicates ////////////////////////////////////////////////////////////////////////////////////////////
  // Allow external callers to identify which state a Repository is in if necessary.

  isLoadingGuess() {
    return false;
  }

  isAbsentGuess() {
    return false;
  }

  isAbsent() {
    return false;
  }

  isLoading() {
    return false;
  }

  isEmpty() {
    return false;
  }

  isPresent() {
    return false;
  }

  isTooLarge() {
    return false;
  }

  isDestroyed() {
    return false;
  }

  // Behavior probe predicates /////////////////////////////////////////////////////////////////////////////////////////
  // Determine specific rendering behavior based on the current state.

  isUndetermined() {
    return false;
  }

  showGitTabInit() {
    return false;
  }

  showGitTabInitInProgress() {
    return false;
  }

  showGitTabLoading() {
    return false;
  }

  showStatusBarTiles() {
    return false;
  }

  hasDirectory() {
    return true;
  }

  // Lifecycle actions /////////////////////////////////////////////////////////////////////////////////////////////////
  // These generally default to rejecting a Promise with an error.

  init() {
    return unsupportedOperationPromise(this, 'init');
  }

  clone(remoteUrl) {
    return unsupportedOperationPromise(this, 'clone');
  }

  destroy() {
    return this.transitionTo('Destroyed');
  }

  refresh() {
    // No-op
  }

  observeFilesystemChange(events) {
    this.repository.refresh();
  }

  updateCommitMessageAfterFileSystemChange() {
    // this is only used in unit tests, we don't need no stinkin coverage
    /* istanbul ignore next */
    this.repository.refresh();
  }

  // Git operations ////////////////////////////////////////////////////////////////////////////////////////////////////
  // These default to rejecting a Promise with an error stating that the operation is not supported in the current
  // state.

  // Staging and unstaging

  stageFiles(paths) {
    return unsupportedOperationPromise(this, 'stageFiles');
  }

  unstageFiles(paths) {
    return unsupportedOperationPromise(this, 'unstageFiles');
  }

  stageFilesFromParentCommit(paths) {
    return unsupportedOperationPromise(this, 'stageFilesFromParentCommit');
  }

  applyPatchToIndex(patch) {
    return unsupportedOperationPromise(this, 'applyPatchToIndex');
  }

  applyPatchToWorkdir(patch) {
    return unsupportedOperationPromise(this, 'applyPatchToWorkdir');
  }

  // Committing

  commit(message, options) {
    return unsupportedOperationPromise(this, 'commit');
  }

  // Merging

  merge(branchName) {
    return unsupportedOperationPromise(this, 'merge');
  }

  abortMerge() {
    return unsupportedOperationPromise(this, 'abortMerge');
  }

  checkoutSide(side, paths) {
    return unsupportedOperationPromise(this, 'checkoutSide');
  }

  mergeFile(oursPath, commonBasePath, theirsPath, resultPath) {
    return unsupportedOperationPromise(this, 'mergeFile');
  }

  writeMergeConflictToIndex(filePath, commonBaseSha, oursSha, theirsSha) {
    return unsupportedOperationPromise(this, 'writeMergeConflictToIndex');
  }

  // Checkout

  checkout(revision, options = {}) {
    return unsupportedOperationPromise(this, 'checkout');
  }

  checkoutPathsAtRevision(paths, revision = 'HEAD') {
    return unsupportedOperationPromise(this, 'checkoutPathsAtRevision');
  }

  // Reset

  undoLastCommit() {
    return unsupportedOperationPromise(this, 'undoLastCommit');
  }

  // Remote interactions

  fetch(branchName) {
    return unsupportedOperationPromise(this, 'fetch');
  }

  pull(branchName) {
    return unsupportedOperationPromise(this, 'pull');
  }

  push(branchName) {
    return unsupportedOperationPromise(this, 'push');
  }

  // Configuration

  setConfig(option, value, { replaceAll } = {}) {
    return unsupportedOperationPromise(this, 'setConfig');
  }

  unsetConfig(option) {
    return unsupportedOperationPromise(this, 'unsetConfig');
  }

  // Direct blob interactions

  createBlob({ filePath, stdin } = {}) {
    return unsupportedOperationPromise(this, 'createBlob');
  }

  expandBlobToFile(absFilePath, sha) {
    return unsupportedOperationPromise(this, 'expandBlobToFile');
  }

  // Discard history

  createDiscardHistoryBlob() {
    return unsupportedOperationPromise(this, 'createDiscardHistoryBlob');
  }

  updateDiscardHistory() {
    return unsupportedOperationPromise(this, 'updateDiscardHistory');
  }

  storeBeforeAndAfterBlobs(filePaths, isSafe, destructiveAction, partialDiscardFilePath = null) {
    return unsupportedOperationPromise(this, 'storeBeforeAndAfterBlobs');
  }

  restoreLastDiscardInTempFiles(isSafe, partialDiscardFilePath = null) {
    return unsupportedOperationPromise(this, 'restoreLastDiscardInTempFiles');
  }

  popDiscardHistory(partialDiscardFilePath = null) {
    return unsupportedOperationPromise(this, 'popDiscardHistory');
  }

  clearDiscardHistory(partialDiscardFilePath = null) {
    return unsupportedOperationPromise(this, 'clearDiscardHistory');
  }

  discardWorkDirChangesForPaths(paths) {
    return unsupportedOperationPromise(this, 'discardWorkDirChangesForPaths');
  }

  // Accessors /////////////////////////////////////////////////////////////////////////////////////////////////////////
  // When possible, these default to "empty" results when invoked in states that don't have information available, or
  // fail in a way that's consistent with the requested information not being found.

  // Index queries

  getStatusBundle() {
    return Promise.resolve({
      stagedFiles: {},
      unstagedFiles: {},
      mergeConflictFiles: {},
      branch: {
        oid: null,
        head: null,
        upstream: null,
        aheadBehind: { ahead: null, behind: null }
      }
    });
  }

  getStatusesForChangedFiles() {
    return Promise.resolve({
      stagedFiles: [],
      unstagedFiles: [],
      mergeConflictFiles: []
    });
  }

  getFilePatchForPath(filePath, options = {}) {
    return Promise.resolve(new _multiFilePatch2.default({}));
  }

  getStagedChangesPatch() {
    return Promise.resolve(new _multiFilePatch2.default({}));
  }

  readFileFromIndex(filePath) {
    return Promise.reject(new Error(`fatal: Path ${filePath} does not exist (neither on disk nor in the index).`));
  }

  // Commit access

  getLastCommit() {
    return Promise.resolve(_commit.nullCommit);
  }

  getRecentCommits() {
    return Promise.resolve([]);
  }

  // Author information

  getAuthors() {
    return Promise.resolve([]);
  }

  // Branches

  getBranches() {
    return Promise.resolve(new _branchSet2.default());
  }

  getHeadDescription() {
    return Promise.resolve('(no repository)');
  }

  // Merging and rebasing status

  isMerging() {
    return Promise.resolve(false);
  }

  isRebasing() {
    return Promise.resolve(false);
  }

  // Remotes

  getRemotes() {
    return Promise.resolve(new _remoteSet2.default([]));
  }

  addRemote() {
    return unsupportedOperationPromise(this, 'addRemote');
  }

  getAheadCount(branchName) {
    return Promise.resolve(null);
  }

  getBehindCount(branchName) {
    return Promise.resolve(null);
  }

  getConfig(option, { local } = {}) {
    return Promise.resolve(null);
  }

  // Direct blob access

  getBlobContents(sha) {
    return Promise.reject(new Error(`fatal: Not a valid object name ${sha}`));
  }

  // Discard history

  hasDiscardHistory(partialDiscardFilePath = null) {
    return false;
  }

  getDiscardHistory(partialDiscardFilePath = null) {
    return [];
  }

  getLastHistorySnapshots(partialDiscardFilePath = null) {
    return null;
  }

  // Atom repo state

  getOperationStates() {
    return _operationStates.nullOperationStates;
  }

  setCommitMessage(message) {
    return unsupportedOperationPromise(this, 'setCommitMessage');
  }

  getCommitMessage() {
    return '';
  }

  fetchCommitMessageTemplate() {
    return unsupportedOperationPromise(this, 'fetchCommitMessageTemplate');
  }

  // Cache

  getCache() {
    return null;
  }

  // Internal //////////////////////////////////////////////////////////////////////////////////////////////////////////
  // Non-delegated methods that provide subclasses with convenient access to Repository properties.

  git() {
    return this.repository.git;
  }

  workdir() {
    return this.repository.getWorkingDirectoryPath();
  }

  // Call methods on the active Repository state, even if the state has transitioned beneath you.
  // Use this to perform operations within `start()` methods to guard against interrupted state transitions.
  current() {
    return this.repository.state;
  }

  // pipeline
  executePipelineAction(...args) {
    return this.repository.executePipelineAction(...args);
  }

  // Return a Promise that will resolve once the state transitions from Loading.
  getLoadPromise() {
    return this.repository.getLoadPromise();
  }

  getRemoteForBranch(branchName) {
    return this.repository.getRemoteForBranch(branchName);
  }

  saveDiscardHistory() {
    return this.repository.saveDiscardHistory();
  }

  // Initiate a transition to another state.
  transitionTo(stateName, ...payload) {
    const StateConstructor = stateConstructors.get(stateName);
    if (StateConstructor === undefined) {
      throw new Error(`Attempt to transition to unrecognized state ${stateName}`);
    }
    return this.repository.transition(this, StateConstructor, ...payload);
  }

  // Event broadcast

  didDestroy() {
    return this.repository.emitter.emit('did-destroy');
  }

  didUpdate() {
    return this.repository.emitter.emit('did-update');
  }

  // Direct git access
  // Non-delegated git operations for internal use within states.

  directResolveDotGitDir() {
    return Promise.resolve(null);
  }

  directGetConfig(key, options = {}) {
    return Promise.resolve(null);
  }

  directGetBlobContents() {
    return Promise.reject(new Error('Not a valid object name'));
  }

  directInit() {
    return Promise.resolve();
  }

  directClone(remoteUrl, options) {
    return Promise.resolve();
  }

  // Deferred operations
  // Direct raw git operations to the current state, even if the state has been changed. Use these methods within
  // start() methods.

  resolveDotGitDir() {
    return this.current().directResolveDotGitDir();
  }

  doInit(workdir) {
    return this.current().directInit();
  }

  doClone(remoteUrl, options) {
    return this.current().directClone(remoteUrl, options);
  }

  // Parse a DiscardHistory payload from the SHA recorded in config.
  async loadHistoryPayload() {
    const historySha = await this.current().directGetConfig('atomGithub.historySha');
    if (!historySha) {
      return {};
    }

    let blob;
    try {
      blob = await this.current().directGetBlobContents(historySha);
    } catch (e) {
      if (/Not a valid object name/.test(e.stdErr)) {
        return {};
      }

      throw e;
    }

    try {
      return JSON.parse(blob);
    } catch (e) {
      return {};
    }
  }

  // Debugging assistance.

  toString() {
    return this.constructor.name;
  }
}

exports.default = State;
function unsupportedOperationPromise(self, opName) {
  return Promise.reject(new Error(`${opName} is not available in ${self} state`));
}
//# sourceMappingURL=data:application/json;charset=utf-8;base64,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