'use strict';

Object.defineProperty(exports, "__esModule", {
  value: true
});

var _extends = Object.assign || function (target) { for (var i = 1; i < arguments.length; i++) { var source = arguments[i]; for (var key in source) { if (Object.prototype.hasOwnProperty.call(source, key)) { target[key] = source[key]; } } } return target; };

var _path = require('path');

var _path2 = _interopRequireDefault(_path);

var _eventKit = require('event-kit');

var _fsExtra = require('fs-extra');

var _fsExtra2 = _interopRequireDefault(_fsExtra);

var _state = require('./state');

var _state2 = _interopRequireDefault(_state);

var _gitShellOutStrategy = require('../../git-shell-out-strategy');

var _workspaceChangeObserver = require('../workspace-change-observer');

var _patch = require('../patch');

var _discardHistory = require('../discard-history');

var _discardHistory2 = _interopRequireDefault(_discardHistory);

var _branch = require('../branch');

var _branch2 = _interopRequireDefault(_branch);

var _author = require('../author');

var _author2 = _interopRequireDefault(_author);

var _branchSet = require('../branch-set');

var _branchSet2 = _interopRequireDefault(_branchSet);

var _remote = require('../remote');

var _remote2 = _interopRequireDefault(_remote);

var _remoteSet = require('../remote-set');

var _remoteSet2 = _interopRequireDefault(_remoteSet);

var _commit = require('../commit');

var _commit2 = _interopRequireDefault(_commit);

var _operationStates = require('../operation-states');

var _operationStates2 = _interopRequireDefault(_operationStates);

var _reporterProxy = require('../../reporter-proxy');

var _helpers = require('../../helpers');

function _interopRequireDefault(obj) { return obj && obj.__esModule ? obj : { default: obj }; }

/**
 * State used when the working directory contains a valid git repository and can be interacted with. Performs
 * actual git operations, caching the results, and broadcasts `onDidUpdate` events when write actions are
 * performed.
 */
class Present extends _state2.default {
  constructor(repository, history) {
    super(repository);

    this.cache = new Cache();

    this.discardHistory = new _discardHistory2.default(this.createBlob.bind(this), this.expandBlobToFile.bind(this), this.mergeFile.bind(this), this.workdir(), { maxHistoryLength: 60 });

    this.operationStates = new _operationStates2.default({ didUpdate: this.didUpdate.bind(this) });

    this.commitMessage = '';
    this.commitMessageTemplate = null;
    this.fetchInitialMessage();

    if (history) {
      this.discardHistory.updateHistory(history);
    }
  }

  setCommitMessage(message, { suppressUpdate } = { suppressUpdate: false }) {
    this.commitMessage = message;
    if (!suppressUpdate) {
      this.didUpdate();
    }
  }

  setCommitMessageTemplate(template) {
    this.commitMessageTemplate = template;
  }

  async fetchInitialMessage() {
    const mergeMessage = await this.repository.getMergeMessage();
    const template = await this.fetchCommitMessageTemplate();
    if (template) {
      this.commitMessageTemplate = template;
    }
    if (mergeMessage) {
      this.setCommitMessage(mergeMessage);
    } else if (template) {
      this.setCommitMessage(template);
    }
  }

  getCommitMessage() {
    return this.commitMessage;
  }

  fetchCommitMessageTemplate() {
    return this.git().fetchCommitMessageTemplate();
  }

  getOperationStates() {
    return this.operationStates;
  }

  isPresent() {
    return true;
  }

  destroy() {
    this.cache.destroy();
    super.destroy();
  }

  showStatusBarTiles() {
    return true;
  }

  acceptInvalidation(spec) {
    this.cache.invalidate(spec());
    this.didUpdate();
  }

  invalidateCacheAfterFilesystemChange(events) {
    const paths = events.map(e => e.special || e.path);
    const keys = new Set();
    for (let i = 0; i < paths.length; i++) {
      const fullPath = paths[i];

      if (fullPath === _workspaceChangeObserver.FOCUS) {
        keys.add(Keys.statusBundle);
        for (const k of Keys.filePatch.eachWithOpts({ staged: false })) {
          keys.add(k);
        }
        continue;
      }

      const includes = (...segments) => fullPath.includes(_path2.default.join(...segments));

      if ((0, _helpers.filePathEndsWith)(fullPath, '.git', 'index')) {
        keys.add(Keys.stagedChanges);
        keys.add(Keys.filePatch.all);
        keys.add(Keys.index.all);
        keys.add(Keys.statusBundle);
        continue;
      }

      if ((0, _helpers.filePathEndsWith)(fullPath, '.git', 'HEAD')) {
        keys.add(Keys.branches);
        keys.add(Keys.lastCommit);
        keys.add(Keys.recentCommits);
        keys.add(Keys.statusBundle);
        keys.add(Keys.headDescription);
        keys.add(Keys.authors);
        continue;
      }

      if (includes('.git', 'refs', 'heads')) {
        keys.add(Keys.branches);
        keys.add(Keys.lastCommit);
        keys.add(Keys.recentCommits);
        keys.add(Keys.headDescription);
        keys.add(Keys.authors);
        continue;
      }

      if (includes('.git', 'refs', 'remotes')) {
        keys.add(Keys.remotes);
        keys.add(Keys.statusBundle);
        keys.add(Keys.headDescription);
        continue;
      }

      if ((0, _helpers.filePathEndsWith)(fullPath, '.git', 'config')) {
        keys.add(Keys.remotes);
        keys.add(Keys.config.all);
        keys.add(Keys.statusBundle);
        continue;
      }

      // File change within the working directory
      const relativePath = _path2.default.relative(this.workdir(), fullPath);
      keys.add(Keys.filePatch.oneWith(relativePath, { staged: false }));
      keys.add(Keys.statusBundle);
    }

    if (keys.size > 0) {
      this.cache.invalidate(Array.from(keys));
      this.didUpdate();
    }
  }

  isCommitMessageClean() {
    if (this.commitMessage.trim() === '') {
      return true;
    } else if (this.commitMessageTemplate) {
      return this.commitMessage === this.commitMessageTemplate;
    }
    return false;
  }

  async updateCommitMessageAfterFileSystemChange(events) {
    for (let i = 0; i < events.length; i++) {
      const event = events[i];

      if (!event.path) {
        continue;
      }

      if ((0, _helpers.filePathEndsWith)(event.path, '.git', 'MERGE_HEAD')) {
        if (event.action === 'created') {
          if (this.isCommitMessageClean()) {
            this.setCommitMessage((await this.repository.getMergeMessage()));
          }
        } else if (event.action === 'deleted') {
          this.setCommitMessage(this.commitMessageTemplate || '');
        }
      }

      if ((0, _helpers.filePathEndsWith)(event.path, '.git', 'config')) {
        // this won't catch changes made to the template file itself...
        const template = await this.fetchCommitMessageTemplate();
        if (template === null) {
          this.setCommitMessage('');
        } else if (this.commitMessageTemplate !== template) {
          this.setCommitMessage(template);
        }
        this.setCommitMessageTemplate(template);
      }
    }
  }

  observeFilesystemChange(events) {
    this.invalidateCacheAfterFilesystemChange(events);
    this.updateCommitMessageAfterFileSystemChange(events);
  }

  refresh() {
    this.cache.clear();
    this.didUpdate();
  }

  init() {
    return super.init().catch(e => {
      e.stdErr = 'This directory already contains a git repository';
      return Promise.reject(e);
    });
  }

  clone() {
    return super.clone().catch(e => {
      e.stdErr = 'This directory already contains a git repository';
      return Promise.reject(e);
    });
  }

  // Git operations ////////////////////////////////////////////////////////////////////////////////////////////////////

  // Staging and unstaging

  stageFiles(paths) {
    return this.invalidate(() => Keys.cacheOperationKeys(paths), () => this.git().stageFiles(paths));
  }

  unstageFiles(paths) {
    return this.invalidate(() => Keys.cacheOperationKeys(paths), () => this.git().unstageFiles(paths));
  }

  stageFilesFromParentCommit(paths) {
    return this.invalidate(() => Keys.cacheOperationKeys(paths), () => this.git().unstageFiles(paths, 'HEAD~'));
  }

  stageFileModeChange(filePath, fileMode) {
    return this.invalidate(() => Keys.cacheOperationKeys([filePath]), () => this.git().stageFileModeChange(filePath, fileMode));
  }

  stageFileSymlinkChange(filePath) {
    return this.invalidate(() => Keys.cacheOperationKeys([filePath]), () => this.git().stageFileSymlinkChange(filePath));
  }

  applyPatchToIndex(multiFilePatch) {
    return this.invalidate(() => Keys.cacheOperationKeys(Array.from(multiFilePatch.getPathSet())), () => {
      const patchStr = multiFilePatch.toString();
      return this.git().applyPatch(patchStr, { index: true });
    });
  }

  applyPatchToWorkdir(multiFilePatch) {
    return this.invalidate(() => Keys.workdirOperationKeys(Array.from(multiFilePatch.getPathSet())), () => {
      const patchStr = multiFilePatch.toString();
      return this.git().applyPatch(patchStr);
    });
  }

  // Committing

  commit(message, options) {
    return this.invalidate(() => [...Keys.headOperationKeys(), ...Keys.filePatch.eachWithOpts({ staged: true }), Keys.headDescription, Keys.branches, Keys.stagedChanges],
    // eslint-disable-next-line no-shadow
    () => this.executePipelineAction('COMMIT', async (message, options = {}) => {
      const coAuthors = options.coAuthors;
      const opts = !coAuthors ? options : _extends({}, options, {
        coAuthors: coAuthors.map(author => {
          return { email: author.getEmail(), name: author.getFullName() };
        })
      });

      await this.git().commit(message, opts);

      // Collect commit metadata metrics
      // note: in GitShellOutStrategy we have counters for all git commands, including `commit`, but here we have
      //       access to additional metadata (unstaged file count) so it makes sense to collect commit events here
      const { unstagedFiles, mergeConflictFiles } = await this.getStatusesForChangedFiles();
      const unstagedCount = Object.keys(_extends({}, unstagedFiles, mergeConflictFiles)).length;
      (0, _reporterProxy.addEvent)('commit', {
        package: 'github',
        partial: unstagedCount > 0,
        amend: !!options.amend,
        coAuthorCount: coAuthors ? coAuthors.length : 0
      });
    }, message, options));
  }

  // Merging

  merge(branchName) {
    return this.invalidate(() => [...Keys.headOperationKeys(), Keys.index.all, Keys.headDescription], () => this.git().merge(branchName));
  }

  abortMerge() {
    return this.invalidate(() => [Keys.statusBundle, Keys.stagedChanges, Keys.filePatch.all, Keys.index.all], async () => {
      await this.git().abortMerge();
      this.setCommitMessage(this.commitMessageTemplate || '');
    });
  }

  checkoutSide(side, paths) {
    return this.git().checkoutSide(side, paths);
  }

  mergeFile(oursPath, commonBasePath, theirsPath, resultPath) {
    return this.git().mergeFile(oursPath, commonBasePath, theirsPath, resultPath);
  }

  writeMergeConflictToIndex(filePath, commonBaseSha, oursSha, theirsSha) {
    return this.invalidate(() => [Keys.statusBundle, Keys.stagedChanges, ...Keys.filePatch.eachWithFileOpts([filePath], [{ staged: false }, { staged: true }]), Keys.index.oneWith(filePath)], () => this.git().writeMergeConflictToIndex(filePath, commonBaseSha, oursSha, theirsSha));
  }

  // Checkout

  checkout(revision, options = {}) {
    return this.invalidate(() => [Keys.stagedChanges, Keys.lastCommit, Keys.recentCommits, Keys.authors, Keys.statusBundle, Keys.index.all, ...Keys.filePatch.eachWithOpts({ staged: true }), Keys.headDescription, Keys.branches],
    // eslint-disable-next-line no-shadow
    () => this.executePipelineAction('CHECKOUT', (revision, options) => {
      return this.git().checkout(revision, options);
    }, revision, options));
  }

  checkoutPathsAtRevision(paths, revision = 'HEAD') {
    return this.invalidate(() => [Keys.statusBundle, Keys.stagedChanges, ...paths.map(fileName => Keys.index.oneWith(fileName)), ...Keys.filePatch.eachWithFileOpts(paths, [{ staged: true }])], () => this.git().checkoutFiles(paths, revision));
  }

  // Reset

  undoLastCommit() {
    return this.invalidate(() => [Keys.stagedChanges, Keys.lastCommit, Keys.recentCommits, Keys.authors, Keys.statusBundle, Keys.index.all, ...Keys.filePatch.eachWithOpts({ staged: true }), Keys.headDescription], async () => {
      try {
        await this.git().reset('soft', 'HEAD~');
        (0, _reporterProxy.addEvent)('undo-last-commit', { package: 'github' });
      } catch (e) {
        if (/unknown revision/.test(e.stdErr)) {
          // Initial commit
          await this.git().deleteRef('HEAD');
        } else {
          throw e;
        }
      }
    });
  }

  // Remote interactions

  fetch(branchName, options = {}) {
    return this.invalidate(() => [Keys.statusBundle, Keys.headDescription],
    // eslint-disable-next-line no-shadow
    () => this.executePipelineAction('FETCH', async branchName => {
      let finalRemoteName = options.remoteName;
      if (!finalRemoteName) {
        const remote = await this.getRemoteForBranch(branchName);
        if (!remote.isPresent()) {
          return null;
        }
        finalRemoteName = remote.getName();
      }
      return this.git().fetch(finalRemoteName, branchName);
    }, branchName));
  }

  pull(branchName, options = {}) {
    return this.invalidate(() => [...Keys.headOperationKeys(), Keys.index.all, Keys.headDescription, Keys.branches],
    // eslint-disable-next-line no-shadow
    () => this.executePipelineAction('PULL', async branchName => {
      let finalRemoteName = options.remoteName;
      if (!finalRemoteName) {
        const remote = await this.getRemoteForBranch(branchName);
        if (!remote.isPresent()) {
          return null;
        }
        finalRemoteName = remote.getName();
      }
      return this.git().pull(finalRemoteName, branchName, options);
    }, branchName));
  }

  push(branchName, options = {}) {
    return this.invalidate(() => {
      const keys = [Keys.statusBundle, Keys.headDescription];

      if (options.setUpstream) {
        keys.push(Keys.branches);
        keys.push(...Keys.config.eachWithSetting(`branch.${branchName}.remote`));
      }

      return keys;
    },
    // eslint-disable-next-line no-shadow
    () => this.executePipelineAction('PUSH', async (branchName, options) => {
      const remote = options.remote || (await this.getRemoteForBranch(branchName));
      return this.git().push(remote.getNameOr('origin'), branchName, options);
    }, branchName, options));
  }

  // Configuration

  setConfig(setting, value, options) {
    return this.invalidate(() => Keys.config.eachWithSetting(setting), () => this.git().setConfig(setting, value, options));
  }

  unsetConfig(setting) {
    return this.invalidate(() => Keys.config.eachWithSetting(setting), () => this.git().unsetConfig(setting));
  }

  // Direct blob interactions

  createBlob(options) {
    return this.git().createBlob(options);
  }

  expandBlobToFile(absFilePath, sha) {
    return this.git().expandBlobToFile(absFilePath, sha);
  }

  // Discard history

  createDiscardHistoryBlob() {
    return this.discardHistory.createHistoryBlob();
  }

  async updateDiscardHistory() {
    const history = await this.loadHistoryPayload();
    this.discardHistory.updateHistory(history);
  }

  async storeBeforeAndAfterBlobs(filePaths, isSafe, destructiveAction, partialDiscardFilePath = null) {
    const snapshots = await this.discardHistory.storeBeforeAndAfterBlobs(filePaths, isSafe, destructiveAction, partialDiscardFilePath);
    if (snapshots) {
      await this.saveDiscardHistory();
    }
    return snapshots;
  }

  restoreLastDiscardInTempFiles(isSafe, partialDiscardFilePath = null) {
    return this.discardHistory.restoreLastDiscardInTempFiles(isSafe, partialDiscardFilePath);
  }

  async popDiscardHistory(partialDiscardFilePath = null) {
    const removed = await this.discardHistory.popHistory(partialDiscardFilePath);
    if (removed) {
      await this.saveDiscardHistory();
    }
  }

  clearDiscardHistory(partialDiscardFilePath = null) {
    this.discardHistory.clearHistory(partialDiscardFilePath);
    return this.saveDiscardHistory();
  }

  discardWorkDirChangesForPaths(paths) {
    return this.invalidate(() => [Keys.statusBundle, ...paths.map(filePath => Keys.filePatch.oneWith(filePath, { staged: false }))], async () => {
      const untrackedFiles = await this.git().getUntrackedFiles();
      const [filesToRemove, filesToCheckout] = partition(paths, f => untrackedFiles.includes(f));
      await this.git().checkoutFiles(filesToCheckout);
      await Promise.all(filesToRemove.map(filePath => {
        const absPath = _path2.default.join(this.workdir(), filePath);
        return _fsExtra2.default.remove(absPath);
      }));
    });
  }

  // Accessors /////////////////////////////////////////////////////////////////////////////////////////////////////////

  // Index queries

  getStatusBundle() {
    return this.cache.getOrSet(Keys.statusBundle, async () => {
      try {
        const bundle = await this.git().getStatusBundle();
        const results = await this.formatChangedFiles(bundle);
        results.branch = bundle.branch;
        if (!results.branch.aheadBehind) {
          results.branch.aheadBehind = { ahead: null, behind: null };
        }
        return results;
      } catch (err) {
        if (err instanceof _gitShellOutStrategy.LargeRepoError) {
          this.transitionTo('TooLarge');
          return {
            branch: {},
            stagedFiles: {},
            unstagedFiles: {},
            mergeConflictFiles: {}
          };
        } else {
          throw err;
        }
      }
    });
  }

  async formatChangedFiles({ changedEntries, untrackedEntries, renamedEntries, unmergedEntries }) {
    const statusMap = {
      A: 'added',
      M: 'modified',
      D: 'deleted',
      U: 'modified',
      T: 'typechange'
    };

    const stagedFiles = {};
    const unstagedFiles = {};
    const mergeConflictFiles = {};

    changedEntries.forEach(entry => {
      if (entry.stagedStatus) {
        stagedFiles[entry.filePath] = statusMap[entry.stagedStatus];
      }
      if (entry.unstagedStatus) {
        unstagedFiles[entry.filePath] = statusMap[entry.unstagedStatus];
      }
    });

    untrackedEntries.forEach(entry => {
      unstagedFiles[entry.filePath] = statusMap.A;
    });

    renamedEntries.forEach(entry => {
      if (entry.stagedStatus === 'R') {
        stagedFiles[entry.filePath] = statusMap.A;
        stagedFiles[entry.origFilePath] = statusMap.D;
      }
      if (entry.unstagedStatus === 'R') {
        unstagedFiles[entry.filePath] = statusMap.A;
        unstagedFiles[entry.origFilePath] = statusMap.D;
      }
      if (entry.stagedStatus === 'C') {
        stagedFiles[entry.filePath] = statusMap.A;
      }
      if (entry.unstagedStatus === 'C') {
        unstagedFiles[entry.filePath] = statusMap.A;
      }
    });

    let statusToHead;

    for (let i = 0; i < unmergedEntries.length; i++) {
      const { stagedStatus, unstagedStatus, filePath } = unmergedEntries[i];
      if (stagedStatus === 'U' || unstagedStatus === 'U' || stagedStatus === 'A' && unstagedStatus === 'A') {
        // Skipping this check here because we only run a single `await`
        // and we only run it in the main, synchronous body of the for loop.
        // eslint-disable-next-line no-await-in-loop
        if (!statusToHead) {
          statusToHead = await this.git().diffFileStatus({ target: 'HEAD' });
        }
        mergeConflictFiles[filePath] = {
          ours: statusMap[stagedStatus],
          theirs: statusMap[unstagedStatus],
          file: statusToHead[filePath] || 'equivalent'
        };
      }
    }

    return { stagedFiles, unstagedFiles, mergeConflictFiles };
  }

  async getStatusesForChangedFiles() {
    const { stagedFiles, unstagedFiles, mergeConflictFiles } = await this.getStatusBundle();
    return { stagedFiles, unstagedFiles, mergeConflictFiles };
  }

  getFilePatchForPath(filePath, { staged } = { staged: false }) {
    return this.cache.getOrSet(Keys.filePatch.oneWith(filePath, { staged }), async () => {
      const diffs = await this.git().getDiffsForFilePath(filePath, { staged });
      return (0, _patch.buildFilePatch)(diffs);
    });
  }

  getStagedChangesPatch() {
    return this.cache.getOrSet(Keys.stagedChanges, () => {
      return this.git().getStagedChangesPatch().then(_patch.buildMultiFilePatch);
    });
  }

  readFileFromIndex(filePath) {
    return this.cache.getOrSet(Keys.index.oneWith(filePath), () => {
      return this.git().readFileFromIndex(filePath);
    });
  }

  // Commit access

  getLastCommit() {
    return this.cache.getOrSet(Keys.lastCommit, async () => {
      const headCommit = await this.git().getHeadCommit();
      return headCommit.unbornRef ? _commit2.default.createUnborn() : new _commit2.default(headCommit);
    });
  }

  getRecentCommits(options) {
    return this.cache.getOrSet(Keys.recentCommits, async () => {
      const commits = await this.git().getCommits(_extends({ ref: 'HEAD' }, options));
      return commits.map(commit => new _commit2.default(commit));
    });
  }

  // Author information

  getAuthors(options) {
    // For now we'll do the naive thing and invalidate anytime HEAD moves. This ensures that we get new authors
    // introduced by newly created commits or pulled commits.
    // This means that we are constantly re-fetching data. If performance becomes a concern we can optimize
    return this.cache.getOrSet(Keys.authors, async () => {
      const authorMap = await this.git().getAuthors(options);
      return Object.keys(authorMap).map(email => new _author2.default(email, authorMap[email]));
    });
  }

  // Branches

  getBranches() {
    return this.cache.getOrSet(Keys.branches, async () => {
      const payloads = await this.git().getBranches();
      const branches = new _branchSet2.default();
      for (const payload of payloads) {
        let upstream = _branch.nullBranch;
        if (payload.upstream) {
          upstream = payload.upstream.remoteName ? _branch2.default.createRemoteTracking(payload.upstream.trackingRef, payload.upstream.remoteName, payload.upstream.remoteRef) : new _branch2.default(payload.upstream.trackingRef);
        }

        let push = upstream;
        if (payload.push) {
          push = payload.push.remoteName ? _branch2.default.createRemoteTracking(payload.push.trackingRef, payload.push.remoteName, payload.push.remoteRef) : new _branch2.default(payload.push.trackingRef);
        }

        branches.add(new _branch2.default(payload.name, upstream, push, payload.head, { sha: payload.sha }));
      }
      return branches;
    });
  }

  getHeadDescription() {
    return this.cache.getOrSet(Keys.headDescription, () => {
      return this.git().describeHead();
    });
  }

  // Merging and rebasing status

  isMerging() {
    return this.git().isMerging(this.repository.getGitDirectoryPath());
  }

  isRebasing() {
    return this.git().isRebasing(this.repository.getGitDirectoryPath());
  }

  // Remotes

  getRemotes() {
    return this.cache.getOrSet(Keys.remotes, async () => {
      const remotesInfo = await this.git().getRemotes();
      return new _remoteSet2.default(remotesInfo.map(({ name, url }) => new _remote2.default(name, url)));
    });
  }

  addRemote(name, url) {
    return this.invalidate(() => [...Keys.config.eachWithSetting(`remote.${name}.url`), ...Keys.config.eachWithSetting(`remote.${name}.fetch`), Keys.remotes],
    // eslint-disable-next-line no-shadow
    () => this.executePipelineAction('ADDREMOTE', async (name, url) => {
      await this.git().addRemote(name, url);
      return new _remote2.default(name, url);
    }, name, url));
  }

  async getAheadCount(branchName) {
    const bundle = await this.getStatusBundle();
    return bundle.branch.aheadBehind.ahead;
  }

  async getBehindCount(branchName) {
    const bundle = await this.getStatusBundle();
    return bundle.branch.aheadBehind.behind;
  }

  getConfig(option, { local } = { local: false }) {
    return this.cache.getOrSet(Keys.config.oneWith(option, { local }), () => {
      return this.git().getConfig(option, { local });
    });
  }

  // Direct blob access

  getBlobContents(sha) {
    return this.cache.getOrSet(Keys.blob(sha), () => {
      return this.git().getBlobContents(sha);
    });
  }

  // Discard history

  hasDiscardHistory(partialDiscardFilePath = null) {
    return this.discardHistory.hasHistory(partialDiscardFilePath);
  }

  getDiscardHistory(partialDiscardFilePath = null) {
    return this.discardHistory.getHistory(partialDiscardFilePath);
  }

  getLastHistorySnapshots(partialDiscardFilePath = null) {
    return this.discardHistory.getLastSnapshots(partialDiscardFilePath);
  }

  // Cache

  getCache() {
    return this.cache;
  }

  invalidate(spec, body) {
    return body().then(result => {
      this.acceptInvalidation(spec);
      return result;
    }, err => {
      this.acceptInvalidation(spec);
      return Promise.reject(err);
    });
  }
}

exports.default = Present;
_state2.default.register(Present);

function partition(array, predicate) {
  const matches = [];
  const nonmatches = [];
  array.forEach(item => {
    if (predicate(item)) {
      matches.push(item);
    } else {
      nonmatches.push(item);
    }
  });
  return [matches, nonmatches];
}

class Cache {
  constructor() {
    this.storage = new Map();
    this.byGroup = new Map();

    this.emitter = new _eventKit.Emitter();
  }

  getOrSet(key, operation) {
    const primary = key.getPrimary();
    const existing = this.storage.get(primary);
    if (existing !== undefined) {
      existing.hits++;
      return existing.promise;
    }

    const created = operation();

    this.storage.set(primary, {
      createdAt: performance.now(),
      hits: 0,
      promise: created
    });

    const groups = key.getGroups();
    for (let i = 0; i < groups.length; i++) {
      const group = groups[i];
      let groupSet = this.byGroup.get(group);
      if (groupSet === undefined) {
        groupSet = new Set();
        this.byGroup.set(group, groupSet);
      }
      groupSet.add(key);
    }

    this.didUpdate();

    return created;
  }

  invalidate(keys) {
    for (let i = 0; i < keys.length; i++) {
      keys[i].removeFromCache(this);
    }

    if (keys.length > 0) {
      this.didUpdate();
    }
  }

  keysInGroup(group) {
    return this.byGroup.get(group) || [];
  }

  removePrimary(primary) {
    this.storage.delete(primary);
    this.didUpdate();
  }

  removeFromGroup(group, key) {
    const groupSet = this.byGroup.get(group);
    groupSet && groupSet.delete(key);
    this.didUpdate();
  }

  [Symbol.iterator]() {
    return this.storage[Symbol.iterator]();
  }

  clear() {
    this.storage.clear();
    this.byGroup.clear();
    this.didUpdate();
  }

  didUpdate() {
    this.emitter.emit('did-update');
  }

  onDidUpdate(callback) {
    return this.emitter.on('did-update', callback);
  }

  destroy() {
    this.emitter.dispose();
  }
}

class CacheKey {
  constructor(primary, groups = []) {
    this.primary = primary;
    this.groups = groups;
  }

  getPrimary() {
    return this.primary;
  }

  getGroups() {
    return this.groups;
  }

  removeFromCache(cache, withoutGroup = null) {
    cache.removePrimary(this.getPrimary());

    const groups = this.getGroups();
    for (let i = 0; i < groups.length; i++) {
      const group = groups[i];
      if (group === withoutGroup) {
        continue;
      }

      cache.removeFromGroup(group, this);
    }
  }

  toString() {
    return `CacheKey(${this.primary})`;
  }
}

class GroupKey {
  constructor(group) {
    this.group = group;
  }

  removeFromCache(cache) {
    for (const matchingKey of cache.keysInGroup(this.group)) {
      matchingKey.removeFromCache(cache, this.group);
    }
  }

  toString() {
    return `GroupKey(${this.group})`;
  }
}

const Keys = {
  statusBundle: new CacheKey('status-bundle'),

  stagedChanges: new CacheKey('staged-changes'),

  filePatch: {
    _optKey: ({ staged }) => {
      return staged ? 's' : 'u';
    },

    oneWith: (fileName, options) => {
      // <-- Keys.filePatch
      const optKey = Keys.filePatch._optKey(options);
      return new CacheKey(`file-patch:${optKey}:${fileName}`, ['file-patch', `file-patch:${optKey}`]);
    },

    eachWithFileOpts: (fileNames, opts) => {
      const keys = [];
      for (let i = 0; i < fileNames.length; i++) {
        for (let j = 0; j < opts.length; j++) {
          keys.push(Keys.filePatch.oneWith(fileNames[i], opts[j]));
        }
      }
      return keys;
    },

    eachWithOpts: (...opts) => opts.map(opt => new GroupKey(`file-patch:${Keys.filePatch._optKey(opt)}`)),

    all: new GroupKey('file-patch')
  },

  index: {
    oneWith: fileName => new CacheKey(`index:${fileName}`, ['index']),

    all: new GroupKey('index')
  },

  lastCommit: new CacheKey('last-commit'),

  recentCommits: new CacheKey('recent-commits'),

  authors: new CacheKey('authors'),

  branches: new CacheKey('branches'),

  headDescription: new CacheKey('head-description'),

  remotes: new CacheKey('remotes'),

  config: {
    _optKey: options => options.local ? 'l' : '',

    oneWith: (setting, options) => {
      const optKey = Keys.config._optKey(options);
      return new CacheKey(`config:${optKey}:${setting}`, ['config', `config:${optKey}`]);
    },

    eachWithSetting: setting => [Keys.config.oneWith(setting, { local: true }), Keys.config.oneWith(setting, { local: false })],

    all: new GroupKey('config')
  },

  blob: {
    oneWith: sha => `blob:${sha}`
  },

  // Common collections of keys and patterns for use with invalidate().

  workdirOperationKeys: fileNames => [Keys.statusBundle, ...Keys.filePatch.eachWithFileOpts(fileNames, [{ staged: false }])],

  cacheOperationKeys: fileNames => [...Keys.workdirOperationKeys(fileNames), ...Keys.filePatch.eachWithFileOpts(fileNames, [{ staged: true }]), ...fileNames.map(Keys.index.oneWith), Keys.stagedChanges],

  headOperationKeys: () => [...Keys.filePatch.eachWithOpts({ staged: true }), Keys.stagedChanges, Keys.lastCommit, Keys.recentCommits, Keys.authors, Keys.statusBundle]
};
//# sourceMappingURL=data:application/json;charset=utf-8;base64,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