'use strict';

Object.defineProperty(exports, "__esModule", {
  value: true
});
class Remote {
  constructor(name, url) {
    this.name = name;
    this.url = url;

    const { isGithubRepo, domain, protocol, owner, repo } = githubInfoFromRemote(url);
    this.githubRepo = isGithubRepo;
    this.domain = domain;
    this.protocol = protocol;
    this.owner = owner;
    this.repo = repo;
  }

  getName() {
    return this.name;
  }

  getUrl() {
    return this.url;
  }

  isGithubRepo() {
    return this.githubRepo;
  }

  getProtocol() {
    return this.protocol;
  }

  getDomain() {
    return this.domain;
  }

  getOwner() {
    return this.owner;
  }

  getRepo() {
    return this.repo;
  }

  getNameOr(fallback) {
    return this.getName();
  }

  getSlug() {
    if (this.owner === null || this.repo === null) {
      return null;
    }

    return `${this.owner}/${this.repo}`;
  }

  isPresent() {
    return true;
  }
}

exports.default = Remote;
function githubInfoFromRemote(remoteUrl) {
  if (!remoteUrl) {
    return {
      isGithubRepo: false,
      domain: null,
      owner: null,
      repo: null
    };
  }

  //             proto         login   domain           owner    repo
  const regex = /(?:(.+):\/\/)?(?:.+@)?(github\.com)[:/]([^/]+)\/(.+)/;
  const match = remoteUrl.match(regex);
  if (match) {
    return {
      isGithubRepo: true,
      protocol: match[1] || 'ssh',
      domain: match[2],
      owner: match[3],
      repo: match[4].replace(/\.git$/, '')
    };
  } else {
    return {
      isGithubRepo: false,
      protocol: null,
      domain: null,
      owner: null,
      repo: null
    };
  }
}

const nullRemote = exports.nullRemote = {
  getName() {
    return '';
  },

  getUrl() {
    return '';
  },

  isGithubRepo() {
    return false;
  },

  getDomain() {
    return null;
  },

  getProtocol() {
    return null;
  },

  getOwner() {
    return null;
  },

  getRepo() {
    return null;
  },

  getNameOr(fallback) {
    return fallback;
  },

  getSlug() {
    return null;
  },

  isPresent() {
    return false;
  }
};
//# sourceMappingURL=data:application/json;charset=utf-8;base64,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