'use strict';

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.NoNewline = exports.Unchanged = exports.Deletion = exports.Addition = undefined;

var _atom = require('atom');

class Region {
  constructor(marker) {
    this.marker = marker;
  }

  getMarker() {
    return this.marker;
  }

  getRange() {
    return this.marker.getRange();
  }

  getStartBufferRow() {
    return this.getRange().start.row;
  }

  getEndBufferRow() {
    return this.getRange().end.row;
  }

  includesBufferRow(row) {
    return this.getRange().intersectsRow(row);
  }

  /*
   * intersectRows breaks a Region into runs of rows that are included in
   * rowSet and rows that are not. For example:
   *  @this Region        row 10-20
   *  @param rowSet       row 11, 12, 13, 17, 19
   *  @param includeGaps  true (whether the result will include gaps or not)
   *  @return an array of regions like this:
   *    (10, gap = true) (11, 12, 13, gap = false) (14, 15, 16, gap = true)
   *    (17, gap = false) (18, gap = true) (19, gap = false) (20, gap = true)
   */
  intersectRows(rowSet, includeGaps) {
    const intersections = [];
    let withinIntersection = false;

    let currentRow = this.getRange().start.row;
    let nextStartRow = currentRow;

    const finishRowRange = isGap => {
      if (isGap && !includeGaps) {
        nextStartRow = currentRow;
        return;
      }

      if (currentRow <= this.getRange().start.row) {
        return;
      }

      intersections.push({
        intersection: _atom.Range.fromObject([[nextStartRow, 0], [currentRow - 1, Infinity]]),
        gap: isGap
      });

      nextStartRow = currentRow;
    };

    while (currentRow <= this.getRange().end.row) {
      if (rowSet.has(currentRow) && !withinIntersection) {
        // One row past the end of a gap. Start of intersecting row range.
        finishRowRange(true);
        withinIntersection = true;
      } else if (!rowSet.has(currentRow) && withinIntersection) {
        // One row past the end of intersecting row range. Start of the next gap.
        finishRowRange(false);
        withinIntersection = false;
      }

      currentRow++;
    }

    finishRowRange(!withinIntersection);
    return intersections;
  }

  isAddition() {
    return false;
  }

  isDeletion() {
    return false;
  }

  isUnchanged() {
    return false;
  }

  isNoNewline() {
    return false;
  }

  getBufferRows() {
    return this.getRange().getRows();
  }

  bufferRowCount() {
    return this.getRange().getRowCount();
  }

  when(callbacks) {
    const callback = callbacks[this.constructor.name.toLowerCase()] || callbacks.default || (() => undefined);
    return callback();
  }

  reMarkOn(markable) {
    this.marker = markable.markRange(this.getRange(), { invalidate: 'never', exclusive: false });
  }

  toStringIn(buffer) {
    const raw = buffer.getTextInRange(this.getRange());
    return this.constructor.origin + raw.replace(/\r?\n/g, '$&' + this.constructor.origin) + buffer.lineEndingForRow(this.getRange().end.row);
  }

  isChange() {
    return true;
  }
}

class Addition extends Region {

  isAddition() {
    return true;
  }

  invertIn(nextBuffer) {
    return new Deletion(nextBuffer.markRange(this.getRange()));
  }
}

exports.Addition = Addition;
Addition.origin = '+';
class Deletion extends Region {

  isDeletion() {
    return true;
  }

  invertIn(nextBuffer) {
    return new Addition(nextBuffer.markRange(this.getRange()));
  }
}

exports.Deletion = Deletion;
Deletion.origin = '-';
class Unchanged extends Region {

  isUnchanged() {
    return true;
  }

  isChange() {
    return false;
  }

  invertIn(nextBuffer) {
    return new Unchanged(nextBuffer.markRange(this.getRange()));
  }
}

exports.Unchanged = Unchanged;
Unchanged.origin = ' ';
class NoNewline extends Region {

  isNoNewline() {
    return true;
  }

  isChange() {
    return false;
  }

  invertIn(nextBuffer) {
    return new NoNewline(nextBuffer.markRange(this.getRange()));
  }
}
exports.NoNewline = NoNewline;
NoNewline.origin = '\\';
//# sourceMappingURL=data:application/json;charset=utf-8;base64,eyJ2ZXJzaW9uIjozLCJzb3VyY2VzIjpbInJlZ2lvbi5qcyJdLCJuYW1lcyI6WyJSZWdpb24iLCJjb25zdHJ1Y3RvciIsIm1hcmtlciIsImdldE1hcmtlciIsImdldFJhbmdlIiwiZ2V0U3RhcnRCdWZmZXJSb3ciLCJzdGFydCIsInJvdyIsImdldEVuZEJ1ZmZlclJvdyIsImVuZCIsImluY2x1ZGVzQnVmZmVyUm93IiwiaW50ZXJzZWN0c1JvdyIsImludGVyc2VjdFJvd3MiLCJyb3dTZXQiLCJpbmNsdWRlR2FwcyIsImludGVyc2VjdGlvbnMiLCJ3aXRoaW5JbnRlcnNlY3Rpb24iLCJjdXJyZW50Um93IiwibmV4dFN0YXJ0Um93IiwiZmluaXNoUm93UmFuZ2UiLCJpc0dhcCIsInB1c2giLCJpbnRlcnNlY3Rpb24iLCJSYW5nZSIsImZyb21PYmplY3QiLCJJbmZpbml0eSIsImdhcCIsImhhcyIsImlzQWRkaXRpb24iLCJpc0RlbGV0aW9uIiwiaXNVbmNoYW5nZWQiLCJpc05vTmV3bGluZSIsImdldEJ1ZmZlclJvd3MiLCJnZXRSb3dzIiwiYnVmZmVyUm93Q291bnQiLCJnZXRSb3dDb3VudCIsIndoZW4iLCJjYWxsYmFja3MiLCJjYWxsYmFjayIsIm5hbWUiLCJ0b0xvd2VyQ2FzZSIsImRlZmF1bHQiLCJ1bmRlZmluZWQiLCJyZU1hcmtPbiIsIm1hcmthYmxlIiwibWFya1JhbmdlIiwiaW52YWxpZGF0ZSIsImV4Y2x1c2l2ZSIsInRvU3RyaW5nSW4iLCJidWZmZXIiLCJyYXciLCJnZXRUZXh0SW5SYW5nZSIsIm9yaWdpbiIsInJlcGxhY2UiLCJsaW5lRW5kaW5nRm9yUm93IiwiaXNDaGFuZ2UiLCJBZGRpdGlvbiIsImludmVydEluIiwibmV4dEJ1ZmZlciIsIkRlbGV0aW9uIiwiVW5jaGFuZ2VkIiwiTm9OZXdsaW5lIl0sIm1hcHBpbmdzIjoiOzs7Ozs7O0FBQUE7O0FBRUEsTUFBTUEsTUFBTixDQUFhO0FBQ1hDLGNBQVlDLE1BQVosRUFBb0I7QUFDbEIsU0FBS0EsTUFBTCxHQUFjQSxNQUFkO0FBQ0Q7O0FBRURDLGNBQVk7QUFDVixXQUFPLEtBQUtELE1BQVo7QUFDRDs7QUFFREUsYUFBVztBQUNULFdBQU8sS0FBS0YsTUFBTCxDQUFZRSxRQUFaLEVBQVA7QUFDRDs7QUFFREMsc0JBQW9CO0FBQ2xCLFdBQU8sS0FBS0QsUUFBTCxHQUFnQkUsS0FBaEIsQ0FBc0JDLEdBQTdCO0FBQ0Q7O0FBRURDLG9CQUFrQjtBQUNoQixXQUFPLEtBQUtKLFFBQUwsR0FBZ0JLLEdBQWhCLENBQW9CRixHQUEzQjtBQUNEOztBQUVERyxvQkFBa0JILEdBQWxCLEVBQXVCO0FBQ3JCLFdBQU8sS0FBS0gsUUFBTCxHQUFnQk8sYUFBaEIsQ0FBOEJKLEdBQTlCLENBQVA7QUFDRDs7QUFFRDs7Ozs7Ozs7OztBQVVBSyxnQkFBY0MsTUFBZCxFQUFzQkMsV0FBdEIsRUFBbUM7QUFDakMsVUFBTUMsZ0JBQWdCLEVBQXRCO0FBQ0EsUUFBSUMscUJBQXFCLEtBQXpCOztBQUVBLFFBQUlDLGFBQWEsS0FBS2IsUUFBTCxHQUFnQkUsS0FBaEIsQ0FBc0JDLEdBQXZDO0FBQ0EsUUFBSVcsZUFBZUQsVUFBbkI7O0FBRUEsVUFBTUUsaUJBQWlCQyxTQUFTO0FBQzlCLFVBQUlBLFNBQVMsQ0FBQ04sV0FBZCxFQUEyQjtBQUN6QkksdUJBQWVELFVBQWY7QUFDQTtBQUNEOztBQUVELFVBQUlBLGNBQWMsS0FBS2IsUUFBTCxHQUFnQkUsS0FBaEIsQ0FBc0JDLEdBQXhDLEVBQTZDO0FBQzNDO0FBQ0Q7O0FBRURRLG9CQUFjTSxJQUFkLENBQW1CO0FBQ2pCQyxzQkFBY0MsWUFBTUMsVUFBTixDQUFpQixDQUFDLENBQUNOLFlBQUQsRUFBZSxDQUFmLENBQUQsRUFBb0IsQ0FBQ0QsYUFBYSxDQUFkLEVBQWlCUSxRQUFqQixDQUFwQixDQUFqQixDQURHO0FBRWpCQyxhQUFLTjtBQUZZLE9BQW5COztBQUtBRixxQkFBZUQsVUFBZjtBQUNELEtBaEJEOztBQWtCQSxXQUFPQSxjQUFjLEtBQUtiLFFBQUwsR0FBZ0JLLEdBQWhCLENBQW9CRixHQUF6QyxFQUE4QztBQUM1QyxVQUFJTSxPQUFPYyxHQUFQLENBQVdWLFVBQVgsS0FBMEIsQ0FBQ0Qsa0JBQS9CLEVBQW1EO0FBQ2pEO0FBQ0FHLHVCQUFlLElBQWY7QUFDQUgsNkJBQXFCLElBQXJCO0FBQ0QsT0FKRCxNQUlPLElBQUksQ0FBQ0gsT0FBT2MsR0FBUCxDQUFXVixVQUFYLENBQUQsSUFBMkJELGtCQUEvQixFQUFtRDtBQUN4RDtBQUNBRyx1QkFBZSxLQUFmO0FBQ0FILDZCQUFxQixLQUFyQjtBQUNEOztBQUVEQztBQUNEOztBQUVERSxtQkFBZSxDQUFDSCxrQkFBaEI7QUFDQSxXQUFPRCxhQUFQO0FBQ0Q7O0FBRURhLGVBQWE7QUFDWCxXQUFPLEtBQVA7QUFDRDs7QUFFREMsZUFBYTtBQUNYLFdBQU8sS0FBUDtBQUNEOztBQUVEQyxnQkFBYztBQUNaLFdBQU8sS0FBUDtBQUNEOztBQUVEQyxnQkFBYztBQUNaLFdBQU8sS0FBUDtBQUNEOztBQUVEQyxrQkFBZ0I7QUFDZCxXQUFPLEtBQUs1QixRQUFMLEdBQWdCNkIsT0FBaEIsRUFBUDtBQUNEOztBQUVEQyxtQkFBaUI7QUFDZixXQUFPLEtBQUs5QixRQUFMLEdBQWdCK0IsV0FBaEIsRUFBUDtBQUNEOztBQUVEQyxPQUFLQyxTQUFMLEVBQWdCO0FBQ2QsVUFBTUMsV0FBV0QsVUFBVSxLQUFLcEMsV0FBTCxDQUFpQnNDLElBQWpCLENBQXNCQyxXQUF0QixFQUFWLEtBQWtESCxVQUFVSSxPQUE1RCxLQUF3RSxNQUFNQyxTQUE5RSxDQUFqQjtBQUNBLFdBQU9KLFVBQVA7QUFDRDs7QUFFREssV0FBU0MsUUFBVCxFQUFtQjtBQUNqQixTQUFLMUMsTUFBTCxHQUFjMEMsU0FBU0MsU0FBVCxDQUFtQixLQUFLekMsUUFBTCxFQUFuQixFQUFvQyxFQUFDMEMsWUFBWSxPQUFiLEVBQXNCQyxXQUFXLEtBQWpDLEVBQXBDLENBQWQ7QUFDRDs7QUFFREMsYUFBV0MsTUFBWCxFQUFtQjtBQUNqQixVQUFNQyxNQUFNRCxPQUFPRSxjQUFQLENBQXNCLEtBQUsvQyxRQUFMLEVBQXRCLENBQVo7QUFDQSxXQUFPLEtBQUtILFdBQUwsQ0FBaUJtRCxNQUFqQixHQUEwQkYsSUFBSUcsT0FBSixDQUFZLFFBQVosRUFBc0IsT0FBTyxLQUFLcEQsV0FBTCxDQUFpQm1ELE1BQTlDLENBQTFCLEdBQ0xILE9BQU9LLGdCQUFQLENBQXdCLEtBQUtsRCxRQUFMLEdBQWdCSyxHQUFoQixDQUFvQkYsR0FBNUMsQ0FERjtBQUVEOztBQUVEZ0QsYUFBVztBQUNULFdBQU8sSUFBUDtBQUNEO0FBdkhVOztBQTBITixNQUFNQyxRQUFOLFNBQXVCeEQsTUFBdkIsQ0FBOEI7O0FBR25DNEIsZUFBYTtBQUNYLFdBQU8sSUFBUDtBQUNEOztBQUVENkIsV0FBU0MsVUFBVCxFQUFxQjtBQUNuQixXQUFPLElBQUlDLFFBQUosQ0FBYUQsV0FBV2IsU0FBWCxDQUFxQixLQUFLekMsUUFBTCxFQUFyQixDQUFiLENBQVA7QUFDRDtBQVRrQzs7UUFBeEJvRCxRLEdBQUFBLFE7QUFBQUEsUSxDQUNKSixNLEdBQVMsRztBQVdYLE1BQU1PLFFBQU4sU0FBdUIzRCxNQUF2QixDQUE4Qjs7QUFHbkM2QixlQUFhO0FBQ1gsV0FBTyxJQUFQO0FBQ0Q7O0FBRUQ0QixXQUFTQyxVQUFULEVBQXFCO0FBQ25CLFdBQU8sSUFBSUYsUUFBSixDQUFhRSxXQUFXYixTQUFYLENBQXFCLEtBQUt6QyxRQUFMLEVBQXJCLENBQWIsQ0FBUDtBQUNEO0FBVGtDOztRQUF4QnVELFEsR0FBQUEsUTtBQUFBQSxRLENBQ0pQLE0sR0FBUyxHO0FBV1gsTUFBTVEsU0FBTixTQUF3QjVELE1BQXhCLENBQStCOztBQUdwQzhCLGdCQUFjO0FBQ1osV0FBTyxJQUFQO0FBQ0Q7O0FBRUR5QixhQUFXO0FBQ1QsV0FBTyxLQUFQO0FBQ0Q7O0FBRURFLFdBQVNDLFVBQVQsRUFBcUI7QUFDbkIsV0FBTyxJQUFJRSxTQUFKLENBQWNGLFdBQVdiLFNBQVgsQ0FBcUIsS0FBS3pDLFFBQUwsRUFBckIsQ0FBZCxDQUFQO0FBQ0Q7QUFibUM7O1FBQXpCd0QsUyxHQUFBQSxTO0FBQUFBLFMsQ0FDSlIsTSxHQUFTLEc7QUFlWCxNQUFNUyxTQUFOLFNBQXdCN0QsTUFBeEIsQ0FBK0I7O0FBR3BDK0IsZ0JBQWM7QUFDWixXQUFPLElBQVA7QUFDRDs7QUFFRHdCLGFBQVc7QUFDVCxXQUFPLEtBQVA7QUFDRDs7QUFFREUsV0FBU0MsVUFBVCxFQUFxQjtBQUNuQixXQUFPLElBQUlHLFNBQUosQ0FBY0gsV0FBV2IsU0FBWCxDQUFxQixLQUFLekMsUUFBTCxFQUFyQixDQUFkLENBQVA7QUFDRDtBQWJtQztRQUF6QnlELFMsR0FBQUEsUztBQUFBQSxTLENBQ0pULE0sR0FBUyxJIiwiZmlsZSI6InJlZ2lvbi5qcyIsInNvdXJjZVJvb3QiOiIvYnVpbGQvYXRvbS9zcmMvYXRvbS0xLjM0LjAvb3V0L2FwcC9ub2RlX21vZHVsZXMvZ2l0aHViL2xpYi9tb2RlbHMvcGF0Y2giLCJzb3VyY2VzQ29udGVudCI6WyJpbXBvcnQge1JhbmdlfSBmcm9tICdhdG9tJztcblxuY2xhc3MgUmVnaW9uIHtcbiAgY29uc3RydWN0b3IobWFya2VyKSB7XG4gICAgdGhpcy5tYXJrZXIgPSBtYXJrZXI7XG4gIH1cblxuICBnZXRNYXJrZXIoKSB7XG4gICAgcmV0dXJuIHRoaXMubWFya2VyO1xuICB9XG5cbiAgZ2V0UmFuZ2UoKSB7XG4gICAgcmV0dXJuIHRoaXMubWFya2VyLmdldFJhbmdlKCk7XG4gIH1cblxuICBnZXRTdGFydEJ1ZmZlclJvdygpIHtcbiAgICByZXR1cm4gdGhpcy5nZXRSYW5nZSgpLnN0YXJ0LnJvdztcbiAgfVxuXG4gIGdldEVuZEJ1ZmZlclJvdygpIHtcbiAgICByZXR1cm4gdGhpcy5nZXRSYW5nZSgpLmVuZC5yb3c7XG4gIH1cblxuICBpbmNsdWRlc0J1ZmZlclJvdyhyb3cpIHtcbiAgICByZXR1cm4gdGhpcy5nZXRSYW5nZSgpLmludGVyc2VjdHNSb3cocm93KTtcbiAgfVxuXG4gIC8qXG4gICAqIGludGVyc2VjdFJvd3MgYnJlYWtzIGEgUmVnaW9uIGludG8gcnVucyBvZiByb3dzIHRoYXQgYXJlIGluY2x1ZGVkIGluXG4gICAqIHJvd1NldCBhbmQgcm93cyB0aGF0IGFyZSBub3QuIEZvciBleGFtcGxlOlxuICAgKiAgQHRoaXMgUmVnaW9uICAgICAgICByb3cgMTAtMjBcbiAgICogIEBwYXJhbSByb3dTZXQgICAgICAgcm93IDExLCAxMiwgMTMsIDE3LCAxOVxuICAgKiAgQHBhcmFtIGluY2x1ZGVHYXBzICB0cnVlICh3aGV0aGVyIHRoZSByZXN1bHQgd2lsbCBpbmNsdWRlIGdhcHMgb3Igbm90KVxuICAgKiAgQHJldHVybiBhbiBhcnJheSBvZiByZWdpb25zIGxpa2UgdGhpczpcbiAgICogICAgKDEwLCBnYXAgPSB0cnVlKSAoMTEsIDEyLCAxMywgZ2FwID0gZmFsc2UpICgxNCwgMTUsIDE2LCBnYXAgPSB0cnVlKVxuICAgKiAgICAoMTcsIGdhcCA9IGZhbHNlKSAoMTgsIGdhcCA9IHRydWUpICgxOSwgZ2FwID0gZmFsc2UpICgyMCwgZ2FwID0gdHJ1ZSlcbiAgICovXG4gIGludGVyc2VjdFJvd3Mocm93U2V0LCBpbmNsdWRlR2Fwcykge1xuICAgIGNvbnN0IGludGVyc2VjdGlvbnMgPSBbXTtcbiAgICBsZXQgd2l0aGluSW50ZXJzZWN0aW9uID0gZmFsc2U7XG5cbiAgICBsZXQgY3VycmVudFJvdyA9IHRoaXMuZ2V0UmFuZ2UoKS5zdGFydC5yb3c7XG4gICAgbGV0IG5leHRTdGFydFJvdyA9IGN1cnJlbnRSb3c7XG5cbiAgICBjb25zdCBmaW5pc2hSb3dSYW5nZSA9IGlzR2FwID0+IHtcbiAgICAgIGlmIChpc0dhcCAmJiAhaW5jbHVkZUdhcHMpIHtcbiAgICAgICAgbmV4dFN0YXJ0Um93ID0gY3VycmVudFJvdztcbiAgICAgICAgcmV0dXJuO1xuICAgICAgfVxuXG4gICAgICBpZiAoY3VycmVudFJvdyA8PSB0aGlzLmdldFJhbmdlKCkuc3RhcnQucm93KSB7XG4gICAgICAgIHJldHVybjtcbiAgICAgIH1cblxuICAgICAgaW50ZXJzZWN0aW9ucy5wdXNoKHtcbiAgICAgICAgaW50ZXJzZWN0aW9uOiBSYW5nZS5mcm9tT2JqZWN0KFtbbmV4dFN0YXJ0Um93LCAwXSwgW2N1cnJlbnRSb3cgLSAxLCBJbmZpbml0eV1dKSxcbiAgICAgICAgZ2FwOiBpc0dhcCxcbiAgICAgIH0pO1xuXG4gICAgICBuZXh0U3RhcnRSb3cgPSBjdXJyZW50Um93O1xuICAgIH07XG5cbiAgICB3aGlsZSAoY3VycmVudFJvdyA8PSB0aGlzLmdldFJhbmdlKCkuZW5kLnJvdykge1xuICAgICAgaWYgKHJvd1NldC5oYXMoY3VycmVudFJvdykgJiYgIXdpdGhpbkludGVyc2VjdGlvbikge1xuICAgICAgICAvLyBPbmUgcm93IHBhc3QgdGhlIGVuZCBvZiBhIGdhcC4gU3RhcnQgb2YgaW50ZXJzZWN0aW5nIHJvdyByYW5nZS5cbiAgICAgICAgZmluaXNoUm93UmFuZ2UodHJ1ZSk7XG4gICAgICAgIHdpdGhpbkludGVyc2VjdGlvbiA9IHRydWU7XG4gICAgICB9IGVsc2UgaWYgKCFyb3dTZXQuaGFzKGN1cnJlbnRSb3cpICYmIHdpdGhpbkludGVyc2VjdGlvbikge1xuICAgICAgICAvLyBPbmUgcm93IHBhc3QgdGhlIGVuZCBvZiBpbnRlcnNlY3Rpbmcgcm93IHJhbmdlLiBTdGFydCBvZiB0aGUgbmV4dCBnYXAuXG4gICAgICAgIGZpbmlzaFJvd1JhbmdlKGZhbHNlKTtcbiAgICAgICAgd2l0aGluSW50ZXJzZWN0aW9uID0gZmFsc2U7XG4gICAgICB9XG5cbiAgICAgIGN1cnJlbnRSb3crKztcbiAgICB9XG5cbiAgICBmaW5pc2hSb3dSYW5nZSghd2l0aGluSW50ZXJzZWN0aW9uKTtcbiAgICByZXR1cm4gaW50ZXJzZWN0aW9ucztcbiAgfVxuXG4gIGlzQWRkaXRpb24oKSB7XG4gICAgcmV0dXJuIGZhbHNlO1xuICB9XG5cbiAgaXNEZWxldGlvbigpIHtcbiAgICByZXR1cm4gZmFsc2U7XG4gIH1cblxuICBpc1VuY2hhbmdlZCgpIHtcbiAgICByZXR1cm4gZmFsc2U7XG4gIH1cblxuICBpc05vTmV3bGluZSgpIHtcbiAgICByZXR1cm4gZmFsc2U7XG4gIH1cblxuICBnZXRCdWZmZXJSb3dzKCkge1xuICAgIHJldHVybiB0aGlzLmdldFJhbmdlKCkuZ2V0Um93cygpO1xuICB9XG5cbiAgYnVmZmVyUm93Q291bnQoKSB7XG4gICAgcmV0dXJuIHRoaXMuZ2V0UmFuZ2UoKS5nZXRSb3dDb3VudCgpO1xuICB9XG5cbiAgd2hlbihjYWxsYmFja3MpIHtcbiAgICBjb25zdCBjYWxsYmFjayA9IGNhbGxiYWNrc1t0aGlzLmNvbnN0cnVjdG9yLm5hbWUudG9Mb3dlckNhc2UoKV0gfHwgY2FsbGJhY2tzLmRlZmF1bHQgfHwgKCgpID0+IHVuZGVmaW5lZCk7XG4gICAgcmV0dXJuIGNhbGxiYWNrKCk7XG4gIH1cblxuICByZU1hcmtPbihtYXJrYWJsZSkge1xuICAgIHRoaXMubWFya2VyID0gbWFya2FibGUubWFya1JhbmdlKHRoaXMuZ2V0UmFuZ2UoKSwge2ludmFsaWRhdGU6ICduZXZlcicsIGV4Y2x1c2l2ZTogZmFsc2V9KTtcbiAgfVxuXG4gIHRvU3RyaW5nSW4oYnVmZmVyKSB7XG4gICAgY29uc3QgcmF3ID0gYnVmZmVyLmdldFRleHRJblJhbmdlKHRoaXMuZ2V0UmFuZ2UoKSk7XG4gICAgcmV0dXJuIHRoaXMuY29uc3RydWN0b3Iub3JpZ2luICsgcmF3LnJlcGxhY2UoL1xccj9cXG4vZywgJyQmJyArIHRoaXMuY29uc3RydWN0b3Iub3JpZ2luKSArXG4gICAgICBidWZmZXIubGluZUVuZGluZ0ZvclJvdyh0aGlzLmdldFJhbmdlKCkuZW5kLnJvdyk7XG4gIH1cblxuICBpc0NoYW5nZSgpIHtcbiAgICByZXR1cm4gdHJ1ZTtcbiAgfVxufVxuXG5leHBvcnQgY2xhc3MgQWRkaXRpb24gZXh0ZW5kcyBSZWdpb24ge1xuICBzdGF0aWMgb3JpZ2luID0gJysnO1xuXG4gIGlzQWRkaXRpb24oKSB7XG4gICAgcmV0dXJuIHRydWU7XG4gIH1cblxuICBpbnZlcnRJbihuZXh0QnVmZmVyKSB7XG4gICAgcmV0dXJuIG5ldyBEZWxldGlvbihuZXh0QnVmZmVyLm1hcmtSYW5nZSh0aGlzLmdldFJhbmdlKCkpKTtcbiAgfVxufVxuXG5leHBvcnQgY2xhc3MgRGVsZXRpb24gZXh0ZW5kcyBSZWdpb24ge1xuICBzdGF0aWMgb3JpZ2luID0gJy0nO1xuXG4gIGlzRGVsZXRpb24oKSB7XG4gICAgcmV0dXJuIHRydWU7XG4gIH1cblxuICBpbnZlcnRJbihuZXh0QnVmZmVyKSB7XG4gICAgcmV0dXJuIG5ldyBBZGRpdGlvbihuZXh0QnVmZmVyLm1hcmtSYW5nZSh0aGlzLmdldFJhbmdlKCkpKTtcbiAgfVxufVxuXG5leHBvcnQgY2xhc3MgVW5jaGFuZ2VkIGV4dGVuZHMgUmVnaW9uIHtcbiAgc3RhdGljIG9yaWdpbiA9ICcgJztcblxuICBpc1VuY2hhbmdlZCgpIHtcbiAgICByZXR1cm4gdHJ1ZTtcbiAgfVxuXG4gIGlzQ2hhbmdlKCkge1xuICAgIHJldHVybiBmYWxzZTtcbiAgfVxuXG4gIGludmVydEluKG5leHRCdWZmZXIpIHtcbiAgICByZXR1cm4gbmV3IFVuY2hhbmdlZChuZXh0QnVmZmVyLm1hcmtSYW5nZSh0aGlzLmdldFJhbmdlKCkpKTtcbiAgfVxufVxuXG5leHBvcnQgY2xhc3MgTm9OZXdsaW5lIGV4dGVuZHMgUmVnaW9uIHtcbiAgc3RhdGljIG9yaWdpbiA9ICdcXFxcJztcblxuICBpc05vTmV3bGluZSgpIHtcbiAgICByZXR1cm4gdHJ1ZTtcbiAgfVxuXG4gIGlzQ2hhbmdlKCkge1xuICAgIHJldHVybiBmYWxzZTtcbiAgfVxuXG4gIGludmVydEluKG5leHRCdWZmZXIpIHtcbiAgICByZXR1cm4gbmV3IE5vTmV3bGluZShuZXh0QnVmZmVyLm1hcmtSYW5nZSh0aGlzLmdldFJhbmdlKCkpKTtcbiAgfVxufVxuIl19