'use strict';

Object.defineProperty(exports, "__esModule", {
  value: true
});

var _atom = require('atom');

var _hunk = require('./hunk');

var _hunk2 = _interopRequireDefault(_hunk);

var _region = require('./region');

function _interopRequireDefault(obj) { return obj && obj.__esModule ? obj : { default: obj }; }

class Patch {
  static createNull() {
    return new NullPatch();
  }

  constructor({ status, hunks, marker }) {
    this.status = status;
    this.hunks = hunks;
    this.marker = marker;

    this.changedLineCount = this.getHunks().reduce((acc, hunk) => acc + hunk.changedLineCount(), 0);
  }

  getStatus() {
    return this.status;
  }

  getMarker() {
    return this.marker;
  }

  getRange() {
    return this.getMarker().getRange();
  }

  getStartRange() {
    const startPoint = this.getMarker().getRange().start;
    return _atom.Range.fromObject([startPoint, startPoint]);
  }

  getHunks() {
    return this.hunks;
  }

  getChangedLineCount() {
    return this.changedLineCount;
  }

  containsRow(row) {
    return this.marker.getRange().intersectsRow(row);
  }

  reMarkOn(markable) {
    this.marker = markable.markRange(this.getRange(), { invalidate: 'never', exclusive: false });
  }

  getMaxLineNumberWidth() {
    const lastHunk = this.hunks[this.hunks.length - 1];
    return lastHunk ? lastHunk.getMaxLineNumberWidth() : 0;
  }

  clone(opts = {}) {
    return new this.constructor({
      status: opts.status !== undefined ? opts.status : this.getStatus(),
      hunks: opts.hunks !== undefined ? opts.hunks : this.getHunks(),
      marker: opts.marker !== undefined ? opts.marker : this.getMarker()
    });
  }

  buildStagePatchForLines(originalBuffer, nextLayeredBuffer, rowSet) {
    const originalBaseOffset = this.getMarker().getRange().start.row;
    const builder = new BufferBuilder(originalBuffer, originalBaseOffset, nextLayeredBuffer);
    const hunks = [];

    let newRowDelta = 0;

    for (const hunk of this.getHunks()) {
      let atLeastOneSelectedChange = false;
      let selectedDeletionRowCount = 0;
      let noNewlineRowCount = 0;

      for (const region of hunk.getRegions()) {
        for (const _ref of region.intersectRows(rowSet, true)) {
          const { intersection, gap } = _ref;

          region.when({
            addition: () => {
              if (gap) {
                // Unselected addition: omit from new buffer
                builder.remove(intersection);
              } else {
                // Selected addition: include in new patch
                atLeastOneSelectedChange = true;
                builder.append(intersection);
                builder.markRegion(intersection, _region.Addition);
              }
            },
            deletion: () => {
              if (gap) {
                // Unselected deletion: convert to context row
                builder.append(intersection);
                builder.markRegion(intersection, _region.Unchanged);
              } else {
                // Selected deletion: include in new patch
                atLeastOneSelectedChange = true;
                builder.append(intersection);
                builder.markRegion(intersection, _region.Deletion);
                selectedDeletionRowCount += intersection.getRowCount();
              }
            },
            unchanged: () => {
              // Untouched context line: include in new patch
              builder.append(intersection);
              builder.markRegion(intersection, _region.Unchanged);
            },
            nonewline: () => {
              builder.append(intersection);
              builder.markRegion(intersection, _region.NoNewline);
              noNewlineRowCount += intersection.getRowCount();
            }
          });
        }
      }

      if (atLeastOneSelectedChange) {
        // Hunk contains at least one selected line

        builder.markHunkRange(hunk.getRange());
        const { regions, marker } = builder.latestHunkWasIncluded();
        const newStartRow = hunk.getNewStartRow() + newRowDelta;
        const newRowCount = marker.getRange().getRowCount() - selectedDeletionRowCount - noNewlineRowCount;

        hunks.push(new _hunk2.default({
          oldStartRow: hunk.getOldStartRow(),
          oldRowCount: hunk.getOldRowCount(),
          newStartRow,
          newRowCount,
          sectionHeading: hunk.getSectionHeading(),
          marker,
          regions
        }));

        newRowDelta += newRowCount - hunk.getNewRowCount();
      } else {
        newRowDelta += hunk.getOldRowCount() - hunk.getNewRowCount();

        builder.latestHunkWasDiscarded();
      }
    }

    const buffer = builder.getBuffer();
    const layers = builder.getLayers();
    const marker = layers.patch.markRange([[0, 0], [buffer.getLastRow() - 1, Infinity]]);

    const wholeFile = rowSet.size === this.changedLineCount;
    const status = this.getStatus() === 'deleted' && !wholeFile ? 'modified' : this.getStatus();
    return this.clone({ hunks, status, marker });
  }

  buildUnstagePatchForLines(originalBuffer, nextLayeredBuffer, rowSet) {
    const originalBaseOffset = this.getMarker().getRange().start.row;
    const builder = new BufferBuilder(originalBuffer, originalBaseOffset, nextLayeredBuffer);
    const hunks = [];
    let newRowDelta = 0;

    for (const hunk of this.getHunks()) {
      let atLeastOneSelectedChange = false;
      let contextRowCount = 0;
      let additionRowCount = 0;
      let deletionRowCount = 0;

      for (const region of hunk.getRegions()) {
        for (const _ref2 of region.intersectRows(rowSet, true)) {
          const { intersection, gap } = _ref2;

          region.when({
            addition: () => {
              if (gap) {
                // Unselected addition: become a context line.
                builder.append(intersection);
                builder.markRegion(intersection, _region.Unchanged);
                contextRowCount += intersection.getRowCount();
              } else {
                // Selected addition: become a deletion.
                atLeastOneSelectedChange = true;
                builder.append(intersection);
                builder.markRegion(intersection, _region.Deletion);
                deletionRowCount += intersection.getRowCount();
              }
            },
            deletion: () => {
              if (gap) {
                // Non-selected deletion: omit from new buffer.
                builder.remove(intersection);
              } else {
                // Selected deletion: becomes an addition
                atLeastOneSelectedChange = true;
                builder.append(intersection);
                builder.markRegion(intersection, _region.Addition);
                additionRowCount += intersection.getRowCount();
              }
            },
            unchanged: () => {
              // Untouched context line: include in new patch.
              builder.append(intersection);
              builder.markRegion(intersection, _region.Unchanged);
              contextRowCount += intersection.getRowCount();
            },
            nonewline: () => {
              // Nonewline marker: include in new patch.
              builder.append(intersection);
              builder.markRegion(intersection, _region.NoNewline);
            }
          });
        }
      }

      if (atLeastOneSelectedChange) {
        // Hunk contains at least one selected line

        builder.markHunkRange(hunk.getRange());
        const { marker, regions } = builder.latestHunkWasIncluded();
        hunks.push(new _hunk2.default({
          oldStartRow: hunk.getNewStartRow(),
          oldRowCount: contextRowCount + deletionRowCount,
          newStartRow: hunk.getNewStartRow() + newRowDelta,
          newRowCount: contextRowCount + additionRowCount,
          sectionHeading: hunk.getSectionHeading(),
          marker,
          regions
        }));
      } else {
        builder.latestHunkWasDiscarded();
      }

      // (contextRowCount + additionRowCount) - (contextRowCount + deletionRowCount)
      newRowDelta += additionRowCount - deletionRowCount;
    }

    const wholeFile = rowSet.size === this.changedLineCount;
    let status = this.getStatus();
    if (this.getStatus() === 'added') {
      status = wholeFile ? 'deleted' : 'modified';
    } else if (this.getStatus() === 'deleted') {
      status = 'added';
    }

    const buffer = builder.getBuffer();
    const layers = builder.getLayers();
    const marker = layers.patch.markRange([[0, 0], [buffer.getLastRow(), Infinity]]);

    return this.clone({ hunks, status, marker });
  }

  getFirstChangeRange() {
    const firstHunk = this.getHunks()[0];
    if (!firstHunk) {
      return _atom.Range.fromObject([[0, 0], [0, 0]]);
    }

    const firstChange = firstHunk.getChanges()[0];
    if (!firstChange) {
      return _atom.Range.fromObject([[0, 0], [0, 0]]);
    }

    const firstRow = firstChange.getStartBufferRow();
    return _atom.Range.fromObject([[firstRow, 0], [firstRow, Infinity]]);
  }

  toStringIn(buffer) {
    return this.getHunks().reduce((str, hunk) => str + hunk.toStringIn(buffer), '');
  }

  isPresent() {
    return true;
  }
}

exports.default = Patch;
class NullPatch {
  constructor() {
    const buffer = new _atom.TextBuffer();
    this.marker = buffer.markRange([[0, 0], [0, 0]]);
  }

  getStatus() {
    return null;
  }

  getMarker() {
    return this.marker;
  }

  getRange() {
    return this.getMarker().getRange();
  }

  getStartRange() {
    return _atom.Range.fromObject([[0, 0], [0, 0]]);
  }

  getHunks() {
    return [];
  }

  getChangedLineCount() {
    return 0;
  }

  containsRow() {
    return false;
  }

  reMarkOn(markable) {
    this.marker = markable.markRange(this.getRange(), { invalidate: 'never', exclusive: false });
  }

  getMaxLineNumberWidth() {
    return 0;
  }

  clone(opts = {}) {
    if (opts.status === undefined && opts.hunks === undefined && opts.marker === undefined) {
      return this;
    } else {
      return new Patch({
        status: opts.status !== undefined ? opts.status : this.getStatus(),
        hunks: opts.hunks !== undefined ? opts.hunks : this.getHunks(),
        marker: opts.marker !== undefined ? opts.marker : this.getMarker()
      });
    }
  }

  buildStagePatchForLines() {
    return this;
  }

  buildUnstagePatchForLines() {
    return this;
  }

  getFirstChangeRange() {
    return _atom.Range.fromObject([[0, 0], [0, 0]]);
  }

  toStringIn() {
    return '';
  }

  isPresent() {
    return false;
  }
}

class BufferBuilder {
  constructor(original, originalBaseOffset, nextLayeredBuffer) {
    this.originalBuffer = original;

    this.buffer = nextLayeredBuffer.buffer;
    this.layers = new Map([[_region.Unchanged, nextLayeredBuffer.layers.unchanged], [_region.Addition, nextLayeredBuffer.layers.addition], [_region.Deletion, nextLayeredBuffer.layers.deletion], [_region.NoNewline, nextLayeredBuffer.layers.noNewline], ['hunk', nextLayeredBuffer.layers.hunk], ['patch', nextLayeredBuffer.layers.patch]]);

    // The ranges provided to builder methods are expected to be valid within the original buffer. Account for
    // the position of the Patch within its original TextBuffer, and any existing content already on the next
    // TextBuffer.
    this.offset = this.buffer.getLastRow() - originalBaseOffset;

    this.hunkBufferText = '';
    this.hunkRowCount = 0;
    this.hunkStartOffset = this.offset;
    this.hunkRegions = [];
    this.hunkRange = null;

    this.lastOffset = 0;
  }

  append(range) {
    this.hunkBufferText += this.originalBuffer.getTextInRange(range) + '\n';
    this.hunkRowCount += range.getRowCount();
  }

  remove(range) {
    this.offset -= range.getRowCount();
  }

  markRegion(range, RegionKind) {
    const finalRange = this.offset !== 0 ? range.translate([this.offset, 0], [this.offset, 0]) : range;

    // Collapse consecutive ranges of the same RegionKind into one continuous region.
    const lastRegion = this.hunkRegions[this.hunkRegions.length - 1];
    if (lastRegion && lastRegion.RegionKind === RegionKind && finalRange.start.row - lastRegion.range.end.row === 1) {
      lastRegion.range.end = finalRange.end;
    } else {
      this.hunkRegions.push({ RegionKind, range: finalRange });
    }
  }

  markHunkRange(range) {
    let finalRange = range;
    if (this.hunkStartOffset !== 0 || this.offset !== 0) {
      finalRange = finalRange.translate([this.hunkStartOffset, 0], [this.offset, 0]);
    }
    this.hunkRange = finalRange;
  }

  latestHunkWasIncluded() {
    this.buffer.append(this.hunkBufferText, { normalizeLineEndings: false });

    const regions = this.hunkRegions.map(({ RegionKind, range }) => {
      return new RegionKind(this.layers.get(RegionKind).markRange(range, { invalidate: 'never', exclusive: false }));
    });

    const marker = this.layers.get('hunk').markRange(this.hunkRange, { invalidate: 'never', exclusive: false });

    this.hunkBufferText = '';
    this.hunkRowCount = 0;
    this.hunkStartOffset = this.offset;
    this.hunkRegions = [];
    this.hunkRange = null;

    return { regions, marker };
  }

  latestHunkWasDiscarded() {
    this.offset -= this.hunkRowCount;

    this.hunkBufferText = '';
    this.hunkRowCount = 0;
    this.hunkStartOffset = this.offset;
    this.hunkRegions = [];
    this.hunkRange = null;

    return { regions: [], marker: null };
  }

  getBuffer() {
    return this.buffer;
  }

  getLayers() {
    return {
      patch: this.layers.get('patch'),
      hunk: this.layers.get('hunk'),
      unchanged: this.layers.get(_region.Unchanged),
      addition: this.layers.get(_region.Addition),
      deletion: this.layers.get(_region.Deletion),
      noNewline: this.layers.get(_region.NoNewline)
    };
  }
}
//# sourceMappingURL=data:application/json;charset=utf-8;base64,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