'use strict';

Object.defineProperty(exports, "__esModule", {
  value: true
});

var _file = require('./file');

var _patch = require('./patch');

var _patch2 = _interopRequireDefault(_patch);

var _helpers = require('../../helpers');

function _interopRequireDefault(obj) { return obj && obj.__esModule ? obj : { default: obj }; }

class FilePatch {
  static createNull() {
    return new this(_file.nullFile, _file.nullFile, _patch2.default.createNull());
  }

  constructor(oldFile, newFile, patch) {
    this.oldFile = oldFile;
    this.newFile = newFile;
    this.patch = patch;
  }

  isPresent() {
    return this.oldFile.isPresent() || this.newFile.isPresent() || this.patch.isPresent();
  }

  getOldFile() {
    return this.oldFile;
  }

  getNewFile() {
    return this.newFile;
  }

  getPatch() {
    return this.patch;
  }

  getMarker() {
    return this.getPatch().getMarker();
  }

  getStartRange() {
    return this.getPatch().getStartRange();
  }

  getOldPath() {
    return this.getOldFile().getPath();
  }

  getNewPath() {
    return this.getNewFile().getPath();
  }

  getOldMode() {
    return this.getOldFile().getMode();
  }

  getNewMode() {
    return this.getNewFile().getMode();
  }

  getOldSymlink() {
    return this.getOldFile().getSymlink();
  }

  getNewSymlink() {
    return this.getNewFile().getSymlink();
  }

  getFirstChangeRange() {
    return this.getPatch().getFirstChangeRange();
  }

  getMaxLineNumberWidth() {
    return this.getPatch().getMaxLineNumberWidth();
  }

  containsRow(row) {
    return this.getPatch().containsRow(row);
  }

  didChangeExecutableMode() {
    if (!this.oldFile.isPresent() || !this.newFile.isPresent()) {
      return false;
    }

    return this.oldFile.isExecutable() && !this.newFile.isExecutable() || !this.oldFile.isExecutable() && this.newFile.isExecutable();
  }

  hasSymlink() {
    return Boolean(this.getOldFile().getSymlink() || this.getNewFile().getSymlink());
  }

  hasTypechange() {
    if (!this.oldFile.isPresent() || !this.newFile.isPresent()) {
      return false;
    }

    return this.oldFile.isSymlink() && !this.newFile.isSymlink() || !this.oldFile.isSymlink() && this.newFile.isSymlink();
  }

  getPath() {
    return this.getOldPath() || this.getNewPath();
  }

  getStatus() {
    return this.getPatch().getStatus();
  }

  getHunks() {
    return this.getPatch().getHunks();
  }

  clone(opts = {}) {
    return new this.constructor(opts.oldFile !== undefined ? opts.oldFile : this.oldFile, opts.newFile !== undefined ? opts.newFile : this.newFile, opts.patch !== undefined ? opts.patch : this.patch);
  }

  buildStagePatchForLines(originalBuffer, nextLayeredBuffer, selectedLineSet) {
    let newFile = this.getNewFile();
    if (this.getStatus() === 'deleted') {
      if (this.patch.getChangedLineCount() === selectedLineSet.size && Array.from(selectedLineSet, row => this.patch.containsRow(row)).every(Boolean)) {
        // Whole file deletion staged.
        newFile = _file.nullFile;
      } else {
        // Partial file deletion, which becomes a modification.
        newFile = this.getOldFile();
      }
    }

    const patch = this.patch.buildStagePatchForLines(originalBuffer, nextLayeredBuffer, selectedLineSet);
    return this.clone({ newFile, patch });
  }

  buildUnstagePatchForLines(originalBuffer, nextLayeredBuffer, selectedLineSet) {
    const nonNullFile = this.getNewFile().isPresent() ? this.getNewFile() : this.getOldFile();
    let oldFile = this.getNewFile();
    let newFile = nonNullFile;

    if (this.getStatus() === 'added') {
      if (selectedLineSet.size === this.patch.getChangedLineCount() && Array.from(selectedLineSet, row => this.patch.containsRow(row)).every(Boolean)) {
        // Ensure that newFile is null if the patch is an addition because we're deleting the entire file from the
        // index. If a symlink was deleted and replaced by a non-symlink file, we don't want the symlink entry to muck
        // up the patch.
        oldFile = nonNullFile;
        newFile = _file.nullFile;
      }
    } else if (this.getStatus() === 'deleted') {
      if (selectedLineSet.size === this.patch.getChangedLineCount() && Array.from(selectedLineSet, row => this.patch.containsRow(row)).every(Boolean)) {
        oldFile = _file.nullFile;
        newFile = nonNullFile;
      }
    }

    const patch = this.patch.buildUnstagePatchForLines(originalBuffer, nextLayeredBuffer, selectedLineSet);
    return this.clone({ oldFile, newFile, patch });
  }

  toStringIn(buffer) {
    if (!this.isPresent()) {
      return '';
    }

    if (this.hasTypechange()) {
      const left = this.clone({
        newFile: _file.nullFile,
        patch: this.getOldSymlink() ? this.getPatch().clone({ status: 'deleted' }) : this.getPatch()
      });

      const right = this.clone({
        oldFile: _file.nullFile,
        patch: this.getNewSymlink() ? this.getPatch().clone({ status: 'added' }) : this.getPatch()
      });

      return left.toStringIn(buffer) + right.toStringIn(buffer);
    } else if (this.getStatus() === 'added' && this.getNewFile().isSymlink()) {
      const symlinkPath = this.getNewSymlink();
      return this.getHeaderString() + `@@ -0,0 +1 @@\n+${symlinkPath}\n\\ No newline at end of file\n`;
    } else if (this.getStatus() === 'deleted' && this.getOldFile().isSymlink()) {
      const symlinkPath = this.getOldSymlink();
      return this.getHeaderString() + `@@ -1 +0,0 @@\n-${symlinkPath}\n\\ No newline at end of file\n`;
    } else {
      return this.getHeaderString() + this.getPatch().toStringIn(buffer);
    }
  }

  getHeaderString() {
    const fromPath = this.getOldPath() || this.getNewPath();
    const toPath = this.getNewPath() || this.getOldPath();
    let header = `diff --git a/${(0, _helpers.toGitPathSep)(fromPath)} b/${(0, _helpers.toGitPathSep)(toPath)}`;
    header += '\n';
    if (this.getStatus() === 'added') {
      header += `new file mode ${this.getNewMode()}`;
      header += '\n';
    } else if (this.getStatus() === 'deleted') {
      header += `deleted file mode ${this.getOldMode()}`;
      header += '\n';
    }
    header += this.getOldPath() ? `--- a/${(0, _helpers.toGitPathSep)(this.getOldPath())}` : '--- /dev/null';
    header += '\n';
    header += this.getNewPath() ? `+++ b/${(0, _helpers.toGitPathSep)(this.getNewPath())}` : '+++ /dev/null';
    header += '\n';
    return header;
  }
}
exports.default = FilePatch;
//# sourceMappingURL=data:application/json;charset=utf-8;base64,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