'use strict';

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.nullOperationStateObserver = exports.FETCH = exports.PULL = exports.PUSH = undefined;

var _eventKit = require('event-kit');

const PUSH = exports.PUSH = {
  getter(o) {
    return o.isPushInProgress();
  }
};

const PULL = exports.PULL = {
  getter(o) {
    return o.isPullInProgress();
  }
};

const FETCH = exports.FETCH = {
  getter(o) {
    return o.isFetchInProgress();
  }
};

// Notify subscibers when a repository completes one or more operations of interest, as observed by its OperationState
// transitioning from `true` to `false`. For exampe, use this to perform actions when a push completes.
class OperationStateObserver {
  constructor(repository, ...operations) {
    this.repository = repository;
    this.operations = new Set(operations);
    this.emitter = new _eventKit.Emitter();

    this.lastStates = new Map();
    for (const operation of this.operations) {
      this.lastStates.set(operation, operation.getter(this.repository.getOperationStates()));
    }

    this.sub = this.repository.onDidUpdate(this.handleUpdate.bind(this));
  }

  onDidComplete(handler) {
    return this.emitter.on('did-complete', handler);
  }

  handleUpdate() {
    let fire = false;
    for (const operation of this.operations) {
      const last = this.lastStates.get(operation);
      const current = operation.getter(this.repository.getOperationStates());
      if (last && !current) {
        fire = true;
      }
      this.lastStates.set(operation, current);
    }
    if (fire) {
      this.emitter.emit('did-complete');
    }
  }

  dispose() {
    this.emitter.dispose();
    this.sub.dispose();
  }
}

exports.default = OperationStateObserver;
const nullOperationStateObserver = exports.nullOperationStateObserver = {
  onDidComplete() {
    return new _eventKit.Disposable();
  },
  dispose() {}
};
//# sourceMappingURL=data:application/json;charset=utf-8;base64,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