"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
class ModelObserver {
  constructor({ fetchData, didUpdate }) {
    this.fetchData = fetchData || (() => {});
    this.didUpdate = didUpdate || (() => {});
    this.activeModel = null;
    this.activeModelData = null;
    this.activeModelUpdateSubscription = null;
    this.inProgress = false;
    this.pending = false;
  }

  setActiveModel(model) {
    if (model !== this.activeModel) {
      if (this.activeModelUpdateSubscription) {
        this.activeModelUpdateSubscription.dispose();
        this.activeModelUpdateSubscription = null;
      }
      this.activeModel = model;
      this.activeModelData = null;
      this.inProgress = false;
      this.pending = false;
      this.didUpdate(model);
      if (model) {
        this.activeModelUpdateSubscription = model.onDidUpdate(() => this.refreshModelData(model));
        return this.refreshModelData(model);
      }
    }
    return null;
  }

  refreshModelData(model = this.activeModel) {
    if (this.inProgress) {
      this.pending = true;
      return null;
    }
    this.lastModelDataRefreshPromise = this._refreshModelData(model);
    return this.lastModelDataRefreshPromise;
  }

  async _refreshModelData(model) {
    try {
      this.inProgress = true;
      const fetchDataPromise = this.fetchData(model);
      this.lastFetchDataPromise = fetchDataPromise;
      const modelData = await fetchDataPromise;
      // Since we re-fetch immediately when the model changes,
      // we need to ensure a fetch for an old active model
      // does not trample the newer fetch for the newer active model.
      if (fetchDataPromise === this.lastFetchDataPromise) {
        this.activeModel = model;
        this.activeModelData = modelData;
        this.didUpdate(model);
      }
    } finally {
      this.inProgress = false;
      if (this.pending) {
        this.pending = false;
        this.refreshModelData();
      }
    }
  }

  getActiveModel() {
    return this.activeModel;
  }

  getActiveModelData() {
    return this.activeModelData;
  }

  getLastModelDataRefreshPromise() {
    return this.lastModelDataRefreshPromise;
  }

  destroy() {
    if (this.activeModelUpdateSubscription) {
      this.activeModelUpdateSubscription.dispose();
    }
  }
}
exports.default = ModelObserver;
//# sourceMappingURL=data:application/json;charset=utf-8;base64,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