'use strict';

Object.defineProperty(exports, "__esModule", {
  value: true
});

var _eventKit = require('event-kit');

var _atom = require('atom');

var _path = require('path');

var _path2 = _interopRequireDefault(_path);

var _eventLogger = require('./event-logger');

var _eventLogger2 = _interopRequireDefault(_eventLogger);

function _interopRequireDefault(obj) { return obj && obj.__esModule ? obj : { default: obj }; }

class FileSystemChangeObserver {
  constructor(repository) {
    this.emitter = new _eventKit.Emitter();
    this.repository = repository;
    this.logger = new _eventLogger2.default('fs watcher');

    this.started = false;
  }

  async start() {
    await this.watchRepository();
    this.started = true;
    return this;
  }

  async destroy() {
    this.started = false;
    this.emitter.dispose();
    await this.stopCurrentFileWatcher();
  }

  isStarted() {
    return this.started;
  }

  didChange(payload) {
    this.emitter.emit('did-change', payload);
  }

  didChangeWorkdirOrHead() {
    this.emitter.emit('did-change-workdir-or-head');
  }

  onDidChange(callback) {
    return this.emitter.on('did-change', callback);
  }

  onDidChangeWorkdirOrHead(callback) {
    return this.emitter.on('did-change-workdir-or-head', callback);
  }

  getRepository() {
    return this.repository;
  }

  async watchRepository() {
    const allPaths = event => {
      const ps = [event.path];
      if (event.oldPath) {
        ps.push(event.oldPath);
      }
      return ps;
    };

    const isNonGitFile = event => allPaths(event).some(eventPath => !eventPath.split(_path2.default.sep).includes('.git'));

    const isWatchedGitFile = event => allPaths(event).some(eventPath => {
      return ['config', 'index', 'HEAD', 'MERGE_HEAD'].includes(_path2.default.basename(eventPath)) || _path2.default.dirname(eventPath).includes(_path2.default.join('.git', 'refs'));
    });

    const handleEvents = events => {
      const filteredEvents = events.filter(e => isNonGitFile(e) || isWatchedGitFile(e));
      if (filteredEvents.length) {
        this.logger.showEvents(filteredEvents);
        this.didChange(filteredEvents);
        const workdirOrHeadEvent = filteredEvents.find(event => {
          return allPaths(event).every(eventPath => !['config', 'index'].includes(_path2.default.basename(eventPath)));
        });
        if (workdirOrHeadEvent) {
          this.logger.showWorkdirOrHeadEvents();
          this.didChangeWorkdirOrHead();
        }
      }
    };

    this.currentFileWatcher = await (0, _atom.watchPath)(this.repository.getWorkingDirectoryPath(), {}, handleEvents);
    this.logger.showStarted(this.repository.getWorkingDirectoryPath(), 'Atom watchPath');
  }

  stopCurrentFileWatcher() {
    if (this.currentFileWatcher) {
      this.currentFileWatcher.dispose();
      this.logger.showStopped();
    }
    return Promise.resolve();
  }
}
exports.default = FileSystemChangeObserver;
//# sourceMappingURL=data:application/json;charset=utf-8;base64,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