'use strict';

Object.defineProperty(exports, "__esModule", {
  value: true
});

var _extends = Object.assign || function (target) { for (var i = 1; i < arguments.length; i++) { var source = arguments[i]; for (var key in source) { if (Object.prototype.hasOwnProperty.call(source, key)) { target[key] = source[key]; } } } return target; };

var _path = require('path');

var _path2 = _interopRequireDefault(_path);

var _os = require('os');

var _os2 = _interopRequireDefault(_os);

var _fsExtra = require('fs-extra');

var _fsExtra2 = _interopRequireDefault(_fsExtra);

var _mkdirp = require('mkdirp');

var _mkdirp2 = _interopRequireDefault(_mkdirp);

var _discardHistoryStores = require('./discard-history-stores');

var _helpers = require('../helpers');

function _interopRequireDefault(obj) { return obj && obj.__esModule ? obj : { default: obj }; }

const emptyFilePath = _path2.default.join(_os2.default.tmpdir(), 'empty-file.txt');
const emptyFilePromise = _fsExtra2.default.writeFile(emptyFilePath, '');

class DiscardHistory {
  constructor(createBlob, expandBlobToFile, mergeFile, workdirPath, { maxHistoryLength } = {}) {
    this.createBlob = createBlob;
    this.expandBlobToFile = expandBlobToFile;
    this.mergeFile = mergeFile;
    this.workdirPath = workdirPath;
    this.partialFileHistory = new _discardHistoryStores.PartialFileDiscardHistory(maxHistoryLength);
    this.wholeFileHistory = new _discardHistoryStores.WholeFileDiscardHistory(maxHistoryLength);
  }

  getLastSnapshots(partialDiscardFilePath = null) {
    if (partialDiscardFilePath) {
      return this.partialFileHistory.getLastSnapshotsForPath(partialDiscardFilePath);
    } else {
      return this.wholeFileHistory.getLastSnapshots();
    }
  }

  getHistory(partialDiscardFilePath = null) {
    if (partialDiscardFilePath) {
      return this.partialFileHistory.getHistoryForPath(partialDiscardFilePath);
    } else {
      return this.wholeFileHistory.getHistory();
    }
  }

  hasHistory(partialDiscardFilePath = null) {
    const history = this.getHistory(partialDiscardFilePath);
    return history.length > 0;
  }

  popHistory(partialDiscardFilePath = null) {
    if (partialDiscardFilePath) {
      return this.partialFileHistory.popHistoryForPath(partialDiscardFilePath);
    } else {
      return this.wholeFileHistory.popHistory();
    }
  }

  clearHistory(partialDiscardFilePath = null) {
    if (partialDiscardFilePath) {
      this.partialFileHistory.clearHistoryForPath(partialDiscardFilePath);
    } else {
      this.wholeFileHistory.clearHistory();
    }
  }

  updateHistory(history) {
    this.partialFileHistory.setHistory(history.partialFileHistory || {});
    this.wholeFileHistory.setHistory(history.wholeFileHistory || []);
  }

  async createHistoryBlob() {
    const histories = {
      wholeFileHistory: this.wholeFileHistory.getHistory(),
      partialFileHistory: this.partialFileHistory.getHistory()
    };
    const historySha = await this.createBlob({ stdin: JSON.stringify(histories) });
    return historySha;
  }

  async storeBeforeAndAfterBlobs(filePaths, isSafe, destructiveAction, partialDiscardFilePath = null) {
    if (partialDiscardFilePath) {
      return await this.storeBlobsForPartialFileHistory(partialDiscardFilePath, isSafe, destructiveAction);
    } else {
      return await this.storeBlobsForWholeFileHistory(filePaths, isSafe, destructiveAction);
    }
  }

  async storeBlobsForPartialFileHistory(filePath, isSafe, destructiveAction) {
    const beforeSha = await this.createBlob({ filePath });
    const isNotSafe = !(await isSafe());
    if (isNotSafe) {
      return null;
    }
    await destructiveAction();
    const afterSha = await this.createBlob({ filePath });
    const snapshots = { beforeSha, afterSha };
    this.partialFileHistory.addHistory(filePath, snapshots);
    return snapshots;
  }

  async storeBlobsForWholeFileHistory(filePaths, isSafe, destructiveAction) {
    const snapshotsByPath = {};
    const beforePromises = filePaths.map(async filePath => {
      snapshotsByPath[filePath] = { beforeSha: await this.createBlob({ filePath }) };
    });
    await Promise.all(beforePromises);
    const isNotSafe = !(await isSafe());
    if (isNotSafe) {
      return null;
    }
    await destructiveAction();
    const afterPromises = filePaths.map(async filePath => {
      snapshotsByPath[filePath].afterSha = await this.createBlob({ filePath });
    });
    await Promise.all(afterPromises);
    this.wholeFileHistory.addHistory(snapshotsByPath);
    return snapshotsByPath;
  }

  async restoreLastDiscardInTempFiles(isSafe, partialDiscardFilePath = null) {
    let lastDiscardSnapshots = this.getLastSnapshots(partialDiscardFilePath);
    if (partialDiscardFilePath) {
      lastDiscardSnapshots = lastDiscardSnapshots ? [lastDiscardSnapshots] : [];
    }
    const tempFolderPaths = await this.expandBlobsToFilesInTempFolder(lastDiscardSnapshots);
    if (!isSafe()) {
      return [];
    }
    return await this.mergeFiles(tempFolderPaths);
  }

  async expandBlobsToFilesInTempFolder(snapshots) {
    const tempFolderPath = await (0, _helpers.getTempDir)({ prefix: 'github-discard-history-' });
    const pathPromises = snapshots.map(async ({ filePath, beforeSha, afterSha }) => {
      const dir = _path2.default.dirname(_path2.default.join(tempFolderPath, filePath));
      await (0, _mkdirp2.default)(dir);
      const theirsPath = !beforeSha ? null : await this.expandBlobToFile(_path2.default.join(tempFolderPath, `${filePath}-before-discard`), beforeSha);
      const commonBasePath = !afterSha ? null : await this.expandBlobToFile(_path2.default.join(tempFolderPath, `${filePath}-after-discard`), afterSha);
      const resultPath = _path2.default.join(tempFolderPath, `~${_path2.default.basename(filePath)}-merge-result`);
      return { filePath, commonBasePath, theirsPath, resultPath, theirsSha: beforeSha, commonBaseSha: afterSha };
    });
    return await Promise.all(pathPromises);
  }

  async mergeFiles(filePaths) {
    const mergeFilePromises = filePaths.map(async (filePathInfo, i) => {
      const { filePath, commonBasePath, theirsPath, resultPath, theirsSha, commonBaseSha } = filePathInfo;
      const currentSha = await this.createBlob({ filePath });
      let mergeResult;
      if (theirsPath && commonBasePath) {
        mergeResult = await this.mergeFile(filePath, commonBasePath, theirsPath, resultPath);
      } else if (!theirsPath && commonBasePath) {
        // deleted file
        const oursSha = await this.createBlob({ filePath });
        if (oursSha === commonBaseSha) {
          // no changes since discard, mark file to be deleted
          mergeResult = { filePath, resultPath: null, deleted: true, conflict: false };
        } else {
          // changes since discard result in conflict
          await _fsExtra2.default.copy(_path2.default.join(this.workdirPath, filePath), resultPath);
          mergeResult = { filePath, resultPath, conflict: true };
        }
      } else if (theirsPath && !commonBasePath) {
        // added file
        const fileDoesExist = await (0, _helpers.fileExists)(_path2.default.join(this.workdirPath, filePath));
        if (!fileDoesExist) {
          await _fsExtra2.default.copy(theirsPath, resultPath);
          mergeResult = { filePath, resultPath, conflict: false };
        } else {
          await emptyFilePromise;
          mergeResult = await this.mergeFile(filePath, emptyFilePath, theirsPath, resultPath);
        }
      } else {
        throw new Error('One of the following must be defined - theirsPath:' + `${theirsPath} or commonBasePath: ${commonBasePath}`);
      }
      return _extends({}, mergeResult, { theirsSha, commonBaseSha, currentSha });
    });
    return await Promise.all(mergeFilePromises);
  }
}
exports.default = DiscardHistory;
//# sourceMappingURL=data:application/json;charset=utf-8;base64,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