'use strict';

Object.defineProperty(exports, "__esModule", {
  value: true
});

var _listSelection = require('./list-selection');

var _listSelection2 = _interopRequireDefault(_listSelection);

function _interopRequireDefault(obj) { return obj && obj.__esModule ? obj : { default: obj }; }

const COPY = Symbol('COPY');

class CompositeListSelection {
  constructor(options) {
    if (options._copy !== COPY) {
      this.keysBySelection = new Map();
      this.selections = [];
      this.idForItem = options.idForItem || (item => item);
      this.resolveNextUpdatePromise = () => {};
      this.activeSelectionIndex = null;

      for (const [key, items] of options.listsByKey) {
        const selection = new _listSelection2.default({ items });
        this.keysBySelection.set(selection, key);
        this.selections.push(selection);

        if (this.activeSelectionIndex === null && selection.getItems().length) {
          this.activeSelectionIndex = this.selections.length - 1;
        }
      }

      if (this.activeSelectionIndex === null) {
        this.activeSelectionIndex = 0;
      }
    } else {
      this.keysBySelection = options.keysBySelection;
      this.selections = options.selections;
      this.idForItem = options.idForItem;
      this.activeSelectionIndex = options.activeSelectionIndex;
      this.resolveNextUpdatePromise = options.resolveNextUpdatePromise;
    }
  }

  copy(options = {}) {
    let selections = [];
    let keysBySelection = new Map();

    if (options.keysBySelection || options.selections) {
      if (!options.keysBySelection || !options.selections) {
        throw new Error('keysBySelection and selection must always be updated simultaneously');
      }

      selections = options.selections;
      keysBySelection = options.keysBySelection;
    } else {
      selections = this.selections;
      keysBySelection = this.keysBySelection;
    }

    return new CompositeListSelection({
      keysBySelection,
      selections,
      activeSelectionIndex: options.activeSelectionIndex !== undefined ? options.activeSelectionIndex : this.activeSelectionIndex,
      idForItem: options.idForItem || this.idForItem,
      resolveNextUpdatePromise: options.resolveNextUpdatePromise || this.resolveNextUpdatePromise,
      _copy: COPY
    });
  }

  updateLists(listsByKey) {
    let isDifferent = false;

    if (listsByKey.length === 0) {
      return this;
    }

    const newKeysBySelection = new Map();
    const newSelections = [];

    for (let i = 0; i < listsByKey.length; i++) {
      const [key, newItems] = listsByKey[i];
      let selection = this.selections[i];

      const oldItems = selection.getItems();
      if (!isDifferent) {
        isDifferent = oldItems.length !== newItems.length || oldItems.some((oldItem, j) => oldItem === newItems[j]);
      }

      const oldHeadItem = selection.getHeadItem();
      selection = selection.setItems(newItems);
      let newHeadItem = null;
      if (oldHeadItem) {
        newHeadItem = newItems.find(item => this.idForItem(item) === this.idForItem(oldHeadItem));
      }
      if (newHeadItem) {
        selection = selection.selectItem(newHeadItem);
      }

      newKeysBySelection.set(selection, key);
      newSelections.push(selection);
    }

    let updated = this.copy({
      keysBySelection: newKeysBySelection,
      selections: newSelections
    });

    if (updated.getActiveSelection().getItems().length === 0) {
      const next = updated.activateNextSelection();
      updated = next !== updated ? next : updated.activatePreviousSelection();
    }

    updated.resolveNextUpdatePromise();
    return updated;
  }

  updateActiveSelection(fn) {
    const oldSelection = this.getActiveSelection();
    const newSelection = fn(oldSelection);
    if (oldSelection === newSelection) {
      return this;
    }

    const key = this.keysBySelection.get(oldSelection);

    const newKeysBySelection = new Map(this.keysBySelection);
    newKeysBySelection.delete(oldSelection);
    newKeysBySelection.set(newSelection, key);

    const newSelections = this.selections.slice();
    newSelections[this.activeSelectionIndex] = newSelection;

    return this.copy({
      keysBySelection: newKeysBySelection,
      selections: newSelections
    });
  }

  getNextUpdatePromise() {
    return new Promise((resolve, reject) => {
      this.resolveNextUpdatePromise = resolve;
    });
  }

  selectFirstNonEmptyList() {
    return this.copy({
      activeSelectionIndex: this.selections.findIndex(selection => selection.getItems().length > 0)
    });
  }

  getActiveListKey() {
    return this.keysBySelection.get(this.getActiveSelection());
  }

  getSelectedItems() {
    return this.getActiveSelection().getSelectedItems();
  }

  getHeadItem() {
    return this.getActiveSelection().getHeadItem();
  }

  getActiveSelection() {
    return this.selections[this.activeSelectionIndex];
  }

  activateSelection(selection) {
    const index = this.selections.indexOf(selection);
    if (index === -1) {
      throw new Error('Selection not found');
    }
    return this.copy({ activeSelectionIndex: index });
  }

  activateNextSelection() {
    for (let i = this.activeSelectionIndex + 1; i < this.selections.length; i++) {
      if (this.selections[i].getItems().length > 0) {
        return this.copy({ activeSelectionIndex: i });
      }
    }
    return this;
  }

  activatePreviousSelection() {
    for (let i = this.activeSelectionIndex - 1; i >= 0; i--) {
      if (this.selections[i].getItems().length > 0) {
        return this.copy({ activeSelectionIndex: i });
      }
    }
    return this;
  }

  activateLastSelection() {
    for (let i = this.selections.length - 1; i >= 0; i--) {
      if (this.selections[i].getItems().length > 0) {
        return this.copy({ activeSelectionIndex: i });
      }
    }
    return this;
  }

  selectItem(item, preserveTail = false) {
    const selection = this.selectionForItem(item);
    if (!selection) {
      throw new Error(`No item found: ${item}`);
    }

    let next = this;
    if (!preserveTail) {
      next = next.activateSelection(selection);
    }
    if (selection === next.getActiveSelection()) {
      next = next.updateActiveSelection(s => s.selectItem(item, preserveTail));
    }
    return next;
  }

  addOrSubtractSelection(item) {
    const selection = this.selectionForItem(item);
    if (!selection) {
      throw new Error(`No item found: ${item}`);
    }

    if (selection === this.getActiveSelection()) {
      return this.updateActiveSelection(s => s.addOrSubtractSelection(item));
    } else {
      return this.activateSelection(selection).updateActiveSelection(s => s.selectItem(item));
    }
  }

  selectAllItems() {
    return this.updateActiveSelection(s => s.selectAllItems());
  }

  selectFirstItem(preserveTail) {
    return this.updateActiveSelection(s => s.selectFirstItem(preserveTail));
  }

  selectLastItem(preserveTail) {
    return this.updateActiveSelection(s => s.selectLastItem(preserveTail));
  }

  coalesce() {
    return this.updateActiveSelection(s => s.coalesce());
  }

  selectionForItem(item) {
    return this.selections.find(selection => selection.getItems().includes(item));
  }

  listKeyForItem(item) {
    return this.keysBySelection.get(this.selectionForItem(item));
  }

  selectNextItem(preserveTail = false) {
    let next = this;
    if (!preserveTail && next.getActiveSelection().getHeadItem() === next.getActiveSelection().getLastItem()) {
      next = next.activateNextSelection();
      if (next !== this) {
        return next.updateActiveSelection(s => s.selectFirstItem());
      } else {
        return next.updateActiveSelection(s => s.selectLastItem());
      }
    } else {
      return next.updateActiveSelection(s => s.selectNextItem(preserveTail));
    }
  }

  selectPreviousItem(preserveTail = false) {
    let next = this;
    if (!preserveTail && next.getActiveSelection().getHeadItem() === next.getActiveSelection().getItems()[0]) {
      next = next.activatePreviousSelection();
      if (next !== this) {
        return next.updateActiveSelection(s => s.selectLastItem());
      } else {
        return next.updateActiveSelection(s => s.selectFirstItem());
      }
    } else {
      return next.updateActiveSelection(s => s.selectPreviousItem(preserveTail));
    }
  }

  findItem(predicate) {
    for (let i = 0; i < this.selections.length; i++) {
      const selection = this.selections[i];
      const key = this.keysBySelection.get(selection);
      const found = selection.getItems().find(item => predicate(item, key));
      if (found !== undefined) {
        return found;
      }
    }
    return null;
  }
}
exports.default = CompositeListSelection;
//# sourceMappingURL=data:application/json;charset=utf-8;base64,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