'use strict';

Object.defineProperty(exports, "__esModule", {
  value: true
});

var _react = require('react');

var _react2 = _interopRequireDefault(_react);

var _propTypes = require('prop-types');

var _propTypes2 = _interopRequireDefault(_propTypes);

var _eventKit = require('event-kit');

var _helpers = require('../helpers');

var _propTypes3 = require('../prop-types');

var _repository = require('../models/repository');

var _repository2 = _interopRequireDefault(_repository);

var _issueishDetailContainer = require('../containers/issueish-detail-container');

var _issueishDetailContainer2 = _interopRequireDefault(_issueishDetailContainer);

var _reporterProxy = require('../reporter-proxy');

function _interopRequireDefault(obj) { return obj && obj.__esModule ? obj : { default: obj }; }

class IssueishDetailItem extends _react.Component {

  static buildURI(host, owner, repo, number, workdir = null) {
    const encodedWorkdir = workdir ? encodeURIComponent(workdir) : '';

    return 'atom-github://issueish/' + encodeURIComponent(host) + '/' + encodeURIComponent(owner) + '/' + encodeURIComponent(repo) + '/' + encodeURIComponent(number) + '?workdir=' + encodedWorkdir;
  }

  constructor(props) {
    super(props);
    (0, _helpers.autobind)(this, 'switchToIssueish', 'handleTitleChanged');

    this.emitter = new _eventKit.Emitter();
    this.title = `${this.props.owner}/${this.props.repo}#${this.props.issueishNumber}`;
    this.hasTerminatedPendingState = false;

    const repository = this.props.workingDirectory === '' ? _repository2.default.absent() : this.props.workdirContextPool.add(this.props.workingDirectory).getRepository();

    this.state = {
      host: this.props.host,
      owner: this.props.owner,
      repo: this.props.repo,
      issueishNumber: this.props.issueishNumber,
      repository
    };

    if (repository.isAbsent()) {
      this.switchToIssueish(this.props.owner, this.props.repo, this.props.issueishNumber);
    }
  }

  render() {
    return _react2.default.createElement(_issueishDetailContainer2.default, {
      host: this.state.host,
      owner: this.state.owner,
      repo: this.state.repo,
      issueishNumber: this.state.issueishNumber,

      repository: this.state.repository,
      loginModel: this.props.loginModel,

      onTitleChange: this.handleTitleChanged,
      switchToIssueish: this.switchToIssueish
    });
  }

  async switchToIssueish(owner, repo, issueishNumber) {
    const pool = this.props.workdirContextPool;
    const prev = {
      owner: this.state.owner,
      repo: this.state.repo,
      issueishNumber: this.state.issueishNumber
    };

    const matchingRepositories = (await Promise.all(pool.withResidentContexts((workdir, context) => {
      const repository = context.getRepository();
      return repository.hasGitHubRemote(this.state.host, owner, repo).then(hasRemote => hasRemote ? repository : null);
    }))).filter(Boolean);
    const nextRepository = matchingRepositories.length === 1 ? matchingRepositories[0] : _repository2.default.absent();

    await new Promise(resolve => {
      this.setState((prevState, props) => {
        if (pool === props.workdirContextPool && prevState.owner === prev.owner && prevState.repo === prev.repo && prevState.issueishNumber === prev.issueishNumber) {
          (0, _reporterProxy.addEvent)('open-issueish-in-pane', { package: 'github', from: 'issueish-link', target: 'current-tab' });
          return {
            owner,
            repo,
            issueishNumber,
            repository: nextRepository
          };
        }

        return {};
      }, resolve);
    });
  }

  handleTitleChanged(title) {
    if (this.title !== title) {
      this.title = title;
      this.emitter.emit('did-change-title', title);
    }
  }

  onDidChangeTitle(cb) {
    return this.emitter.on('did-change-title', cb);
  }

  terminatePendingState() {
    if (!this.hasTerminatedPendingState) {
      this.emitter.emit('did-terminate-pending-state');
      this.hasTerminatedPendingState = true;
    }
  }

  onDidTerminatePendingState(callback) {
    return this.emitter.on('did-terminate-pending-state', callback);
  }

  onDidDestroy(callback) {
    return this.emitter.on('did-destroy', callback);
  }

  serialize() {
    return {
      uri: this.getURI(),
      deserializer: 'IssueishDetailItem'
    };
  }

  getTitle() {
    return this.title;
  }
}
exports.default = IssueishDetailItem;
IssueishDetailItem.propTypes = {
  host: _propTypes2.default.string.isRequired,
  owner: _propTypes2.default.string.isRequired,
  repo: _propTypes2.default.string.isRequired,
  issueishNumber: _propTypes2.default.number.isRequired,

  workingDirectory: _propTypes2.default.string.isRequired,
  workdirContextPool: _propTypes3.WorkdirContextPoolPropType.isRequired,
  loginModel: _propTypes3.GithubLoginModelPropType.isRequired
};
IssueishDetailItem.uriPattern = 'atom-github://issueish/{host}/{owner}/{repo}/{issueishNumber}?workdir={workingDirectory}';
//# sourceMappingURL=data:application/json;charset=utf-8;base64,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