'use strict';

Object.defineProperty(exports, "__esModule", {
  value: true
});

var _extends = Object.assign || function (target) { for (var i = 1; i < arguments.length; i++) { var source = arguments[i]; for (var key in source) { if (Object.prototype.hasOwnProperty.call(source, key)) { target[key] = source[key]; } } } return target; };

var _eventKit = require('event-kit');

var _path = require('path');

var _path2 = _interopRequireDefault(_path);

var _fsExtra = require('fs-extra');

var _fsExtra2 = _interopRequireDefault(_fsExtra);

var _react = require('react');

var _react2 = _interopRequireDefault(_react);

var _reactDom = require('react-dom');

var _reactDom2 = _interopRequireDefault(_reactDom);

var _helpers = require('./helpers');

var _workdirCache = require('./models/workdir-cache');

var _workdirCache2 = _interopRequireDefault(_workdirCache);

var _workdirContext = require('./models/workdir-context');

var _workdirContext2 = _interopRequireDefault(_workdirContext);

var _workdirContextPool = require('./models/workdir-context-pool');

var _workdirContextPool2 = _interopRequireDefault(_workdirContextPool);

var _repository = require('./models/repository');

var _repository2 = _interopRequireDefault(_repository);

var _styleCalculator = require('./models/style-calculator');

var _styleCalculator2 = _interopRequireDefault(_styleCalculator);

var _githubLoginModel = require('./models/github-login-model');

var _githubLoginModel2 = _interopRequireDefault(_githubLoginModel);

var _rootController = require('./controllers/root-controller');

var _rootController2 = _interopRequireDefault(_rootController);

var _stubItem = require('./items/stub-item');

var _stubItem2 = _interopRequireDefault(_stubItem);

var _switchboard = require('./switchboard');

var _switchboard2 = _interopRequireDefault(_switchboard);

var _yardstick = require('./yardstick');

var _yardstick2 = _interopRequireDefault(_yardstick);

var _gitTimingsView = require('./views/git-timings-view');

var _gitTimingsView2 = _interopRequireDefault(_gitTimingsView);

var _contextMenuInterceptor = require('./context-menu-interceptor');

var _contextMenuInterceptor2 = _interopRequireDefault(_contextMenuInterceptor);

var _asyncQueue = require('./async-queue');

var _asyncQueue2 = _interopRequireDefault(_asyncQueue);

var _workerManager = require('./worker-manager');

var _workerManager2 = _interopRequireDefault(_workerManager);

var _getRepoPipelineManager = require('./get-repo-pipeline-manager');

var _getRepoPipelineManager2 = _interopRequireDefault(_getRepoPipelineManager);

var _reporterProxy = require('./reporter-proxy');

function _interopRequireDefault(obj) { return obj && obj.__esModule ? obj : { default: obj }; }

const defaultState = {
  newProject: true
};

class GithubPackage {
  constructor({
    workspace, project, commandRegistry, notificationManager, tooltips, styles, grammars,
    keymaps, config, deserializers,
    confirm, getLoadSettings,
    configDirPath,
    renderFn, loginModel
  }) {
    (0, _helpers.autobind)(this, 'consumeStatusBar', 'createGitTimingsView', 'createIssueishPaneItemStub', 'createDockItemStub', 'createFilePatchControllerStub', 'destroyGitTabItem', 'destroyGithubTabItem', 'createRepositoryForProjectPath', 'cloneRepositoryForProjectPath', 'getRepositoryForWorkdir', 'scheduleActiveContextUpdate');

    this.workspace = workspace;
    this.project = project;
    this.commandRegistry = commandRegistry;
    this.deserializers = deserializers;
    this.notificationManager = notificationManager;
    this.tooltips = tooltips;
    this.config = config;
    this.styles = styles;
    this.grammars = grammars;
    this.keymaps = keymaps;
    this.configPath = _path2.default.join(configDirPath, 'github.cson');

    this.styleCalculator = new _styleCalculator2.default(this.styles, this.config);
    this.confirm = confirm;
    this.startOpen = false;
    this.activated = false;

    const criteria = {
      projectPathCount: this.project.getPaths().length,
      initPathCount: (getLoadSettings().initialPaths || []).length
    };

    this.pipelineManager = (0, _getRepoPipelineManager2.default)({ confirm, notificationManager, workspace });

    this.activeContextQueue = new _asyncQueue2.default();
    this.guessedContext = _workdirContext2.default.guess(criteria, this.pipelineManager);
    this.activeContext = this.guessedContext;
    this.workdirCache = new _workdirCache2.default();
    this.contextPool = new _workdirContextPool2.default({
      window,
      workspace,
      promptCallback: query => this.controller.promptForCredentials(query),
      pipelineManager: this.pipelineManager
    });

    this.switchboard = new _switchboard2.default();

    this.loginModel = loginModel || new _githubLoginModel2.default();
    this.renderFn = renderFn || ((component, node, callback) => {
      return _reactDom2.default.render(component, node, callback);
    });

    // Handle events from all resident contexts.
    this.subscriptions = new _eventKit.CompositeDisposable(this.contextPool.onDidChangeWorkdirOrHead(context => {
      this.refreshAtomGitRepository(context.getWorkingDirectory());
    }), this.contextPool.onDidUpdateRepository(context => {
      this.switchboard.didUpdateRepository(context.getRepository());
    }), this.contextPool.onDidDestroyRepository(context => {
      if (context === this.activeContext) {
        this.setActiveContext(_workdirContext2.default.absent({ pipelineManager: this.pipelineManager }));
      }
    }), _contextMenuInterceptor2.default);

    this.setupYardstick();
  }

  setupYardstick() {
    const stagingSeries = ['stageLine', 'stageHunk', 'unstageLine', 'unstageHunk'];

    this.subscriptions.add(
    // Staging and unstaging operations
    this.switchboard.onDidBeginStageOperation(payload => {
      if (payload.stage && payload.line) {
        _yardstick2.default.begin('stageLine');
      } else if (payload.stage && payload.hunk) {
        _yardstick2.default.begin('stageHunk');
      } else if (payload.stage && payload.file) {
        _yardstick2.default.begin('stageFile');
      } else if (payload.stage && payload.mode) {
        _yardstick2.default.begin('stageMode');
      } else if (payload.stage && payload.symlink) {
        _yardstick2.default.begin('stageSymlink');
      } else if (payload.unstage && payload.line) {
        _yardstick2.default.begin('unstageLine');
      } else if (payload.unstage && payload.hunk) {
        _yardstick2.default.begin('unstageHunk');
      } else if (payload.unstage && payload.file) {
        _yardstick2.default.begin('unstageFile');
      } else if (payload.unstage && payload.mode) {
        _yardstick2.default.begin('unstageMode');
      } else if (payload.unstage && payload.symlink) {
        _yardstick2.default.begin('unstageSymlink');
      }
    }), this.switchboard.onDidUpdateRepository(() => {
      _yardstick2.default.mark(stagingSeries, 'update-repository');
    }), this.switchboard.onDidFinishRender(context => {
      if (context === 'RootController.showFilePatchForPath') {
        _yardstick2.default.finish(stagingSeries);
      }
    }),

    // Active context changes
    this.switchboard.onDidScheduleActiveContextUpdate(() => {
      _yardstick2.default.begin('activeContextChange');
    }), this.switchboard.onDidBeginActiveContextUpdate(() => {
      _yardstick2.default.mark('activeContextChange', 'queue-wait');
    }), this.switchboard.onDidFinishContextChangeRender(() => {
      _yardstick2.default.mark('activeContextChange', 'render');
    }), this.switchboard.onDidFinishActiveContextUpdate(() => {
      _yardstick2.default.finish('activeContextChange');
    }));
  }

  async activate(state = {}) {
    this.savedState = _extends({}, defaultState, state);

    const firstRun = !(await (0, _helpers.fileExists)(this.configPath));
    const newProject = this.savedState.firstRun !== undefined ? this.savedState.firstRun : this.savedState.newProject;

    this.startOpen = firstRun || newProject;
    this.startRevealed = firstRun && !this.config.get('welcome.showOnStartup');

    if (firstRun) {
      await _fsExtra2.default.writeFile(this.configPath, '# Store non-visible GitHub package state.\n', { encoding: 'utf8' });
    }

    const hasSelectedFiles = event => {
      return !!event.target.closest('.github-FilePatchListView').querySelector('.is-selected');
    };

    this.subscriptions.add(this.project.onDidChangePaths(this.scheduleActiveContextUpdate), this.workspace.getCenter().onDidChangeActivePaneItem(this.scheduleActiveContextUpdate), this.styleCalculator.startWatching('github-package-styles', ['editor.fontSize', 'editor.fontFamily', 'editor.lineHeight', 'editor.tabLength'], config => `
          .github-FilePatchView {
            font-size: 1.1em;
          }

          .github-HunkView-line {
            font-size: ${config.get('editor.fontSize')}px;
            font-family: ${config.get('editor.fontFamily')};
            line-height: ${config.get('editor.lineHeight')};
            tab-size: ${config.get('editor.tabLength')}
          }
        `), atom.contextMenu.add({
      '.github-UnstagedChanges .github-FilePatchListView': [{
        label: 'Stage',
        command: 'core:confirm',
        shouldDisplay: hasSelectedFiles
      }, {
        type: 'separator',
        shouldDisplay: hasSelectedFiles
      }, {
        label: 'Discard Changes',
        command: 'github:discard-changes-in-selected-files',
        shouldDisplay: hasSelectedFiles
      }],
      '.github-StagedChanges .github-FilePatchListView': [{
        label: 'Unstage',
        command: 'core:confirm',
        shouldDisplay: hasSelectedFiles
      }],
      '.github-MergeConflictPaths .github-FilePatchListView': [{
        label: 'Stage',
        command: 'core:confirm',
        shouldDisplay: hasSelectedFiles
      }, {
        type: 'separator',
        shouldDisplay: hasSelectedFiles
      }, {
        label: 'Resolve File As Ours',
        command: 'github:resolve-file-as-ours',
        shouldDisplay: hasSelectedFiles
      }, {
        label: 'Resolve File As Theirs',
        command: 'github:resolve-file-as-theirs',
        shouldDisplay: hasSelectedFiles
      }]
    }));

    this.activated = true;
    this.scheduleActiveContextUpdate(this.savedState);
    this.rerender();
  }

  serialize() {
    const activeRepository = this.getActiveRepository();
    const activeRepositoryPath = activeRepository ? activeRepository.getWorkingDirectoryPath() : null;

    return {
      activeRepositoryPath,
      newProject: false
    };
  }

  rerender(callback) {
    if (this.workspace.isDestroyed()) {
      return;
    }

    if (!this.activated) {
      return;
    }

    if (!this.element) {
      this.element = document.createElement('div');
      this.subscriptions.add(new _eventKit.Disposable(() => {
        _reactDom2.default.unmountComponentAtNode(this.element);
        delete this.element;
      }));
    }

    this.renderFn(_react2.default.createElement(_rootController2.default, {
      ref: c => {
        this.controller = c;
      },
      workspace: this.workspace,
      deserializers: this.deserializers,
      commandRegistry: this.commandRegistry,
      notificationManager: this.notificationManager,
      tooltips: this.tooltips,
      grammars: this.grammars,
      keymaps: this.keymaps,
      config: this.config,
      project: this.project,
      confirm: this.confirm,
      workdirContextPool: this.contextPool,
      loginModel: this.loginModel,
      repository: this.getActiveRepository(),
      resolutionProgress: this.getActiveResolutionProgress(),
      statusBar: this.statusBar,
      createRepositoryForProjectPath: this.createRepositoryForProjectPath,
      cloneRepositoryForProjectPath: this.cloneRepositoryForProjectPath,
      switchboard: this.switchboard,
      startOpen: this.startOpen,
      startRevealed: this.startRevealed,
      removeFilePatchItem: this.removeFilePatchItem
    }), this.element, callback);
  }

  async deactivate() {
    this.subscriptions.dispose();
    this.contextPool.clear();
    _workerManager2.default.reset(false);
    if (this.guessedContext) {
      this.guessedContext.destroy();
      this.guessedContext = null;
    }
    await _yardstick2.default.flush();
  }

  consumeStatusBar(statusBar) {
    this.statusBar = statusBar;
    this.rerender();
  }

  consumeReporter(reporter) {
    _reporterProxy.reporterProxy.setReporter(reporter);
  }

  createGitTimingsView() {
    return _stubItem2.default.create('git-timings-view', {
      title: 'GitHub Package Timings View'
    }, _gitTimingsView2.default.buildURI());
  }

  createIssueishPaneItemStub({ uri }) {
    return _stubItem2.default.create('issueish-detail-item', {
      title: 'Issueish'
    }, uri);
  }

  createDockItemStub({ uri }) {
    let item;
    switch (uri) {
      // always return an empty stub
      // but only set it as the active item for a tab type
      // if it doesn't already exist
      case 'atom-github://dock-item/git':
        item = this.createGitStub(uri);
        this.gitTabStubItem = this.gitTabStubItem || item;
        break;
      case 'atom-github://dock-item/github':
        item = this.createGitHubStub(uri);
        this.githubTabStubItem = this.githubTabStubItem || item;
        break;
      default:
        throw new Error(`Invalid DockItem stub URI: ${uri}`);
    }

    if (this.controller) {
      this.rerender();
    }
    return item;
  }

  createGitStub(uri) {
    return _stubItem2.default.create('git', {
      title: 'Git'
    }, uri);
  }

  createGitHubStub(uri) {
    return _stubItem2.default.create('github', {
      title: 'GitHub'
    }, uri);
  }

  createFilePatchControllerStub({ uri } = {}) {
    const item = _stubItem2.default.create('git-file-patch-controller', {
      title: 'Diff'
    }, uri);
    if (this.controller) {
      this.rerender();
    }
    return item;
  }

  createCommitPreviewStub({ uri }) {
    const item = _stubItem2.default.create('git-commit-preview', {
      title: 'Commit preview'
    }, uri);
    if (this.controller) {
      this.rerender();
    }
    return item;
  }

  destroyGitTabItem() {
    if (this.gitTabStubItem) {
      this.gitTabStubItem.destroy();
      this.gitTabStubItem = null;
      if (this.controller) {
        this.rerender();
      }
    }
  }

  destroyGithubTabItem() {
    if (this.githubTabStubItem) {
      this.githubTabStubItem.destroy();
      this.githubTabStubItem = null;
      if (this.controller) {
        this.rerender();
      }
    }
  }

  async createRepositoryForProjectPath(projectPath) {
    await _fsExtra2.default.mkdirs(projectPath);

    const repository = this.contextPool.add(projectPath).getRepository();
    await repository.init();
    this.workdirCache.invalidate();

    if (!this.project.contains(projectPath)) {
      this.project.addPath(projectPath);
    }

    await this.scheduleActiveContextUpdate();
  }

  async cloneRepositoryForProjectPath(remoteUrl, projectPath) {
    const context = this.contextPool.getContext(projectPath);
    let repository;
    if (context.isPresent()) {
      repository = context.getRepository();
      await repository.clone(remoteUrl);
      repository.destroy();
    } else {
      repository = new _repository2.default(projectPath, null, { pipelineManager: this.pipelineManager });
      await repository.clone(remoteUrl);
    }

    this.workdirCache.invalidate();

    this.project.addPath(projectPath);

    await this.scheduleActiveContextUpdate();
  }

  getRepositoryForWorkdir(projectPath) {
    const loadingGuessRepo = _repository2.default.loadingGuess({ pipelineManager: this.pipelineManager });
    return this.guessedContext ? loadingGuessRepo : this.contextPool.getContext(projectPath).getRepository();
  }

  getActiveWorkdir() {
    return this.activeContext.getWorkingDirectory();
  }

  getActiveRepository() {
    return this.activeContext.getRepository();
  }

  getActiveResolutionProgress() {
    return this.activeContext.getResolutionProgress();
  }

  getContextPool() {
    return this.contextPool;
  }

  getSwitchboard() {
    return this.switchboard;
  }

  async scheduleActiveContextUpdate(savedState = {}) {
    this.switchboard.didScheduleActiveContextUpdate();
    await this.activeContextQueue.push(this.updateActiveContext.bind(this, savedState), { parallel: false });
  }

  /**
   * Derive the git working directory context that should be used for the package's git operations based on the current
   * state of the Atom workspace. In priority, this prefers:
   *
   * - A git working directory that contains the active pane item in the workspace's center.
   * - A git working directory corresponding to a single Project.
   * - When initially activating the package, the working directory that was active when the package was last
   *   serialized.
   * - The current context, unchanged, which may be a `NullWorkdirContext`.
   *
   * First updates the pool of resident contexts to match all git working directories that correspond to open
   * projects and pane items.
   */
  async getNextContext(savedState) {
    const workdirs = new Set((await Promise.all(this.project.getPaths().map(async projectPath => {
      const workdir = await this.workdirCache.find(projectPath);
      return workdir || projectPath;
    }))));

    const fromPaneItem = async maybeItem => {
      const itemPath = pathForPaneItem(maybeItem);

      if (!itemPath) {
        return {};
      }

      const itemWorkdir = await this.workdirCache.find(itemPath);

      if (itemWorkdir && !this.project.contains(itemPath)) {
        workdirs.add(itemWorkdir);
      }

      return { itemPath, itemWorkdir };
    };

    const active = await fromPaneItem(this.workspace.getCenter().getActivePaneItem());

    this.contextPool.set(workdirs, savedState);

    if (active.itemPath) {
      // Prefer an active item
      return this.contextPool.getContext(active.itemWorkdir || active.itemPath);
    }

    if (this.project.getPaths().length === 1) {
      // Single project
      const projectPath = this.project.getPaths()[0];
      const activeWorkingDir = await this.workdirCache.find(projectPath);
      return this.contextPool.getContext(activeWorkingDir || projectPath);
    }

    if (this.project.getPaths().length === 0 && !this.activeContext.getRepository().isUndetermined()) {
      // No projects. Revert to the absent context unless we've guessed that more projects are on the way.
      return _workdirContext2.default.absent({ pipelineManager: this.pipelineManager });
    }

    // Restore models from saved state. Will return a NullWorkdirContext if this path is not presently
    // resident in the pool.
    const savedWorkingDir = savedState.activeRepositoryPath;
    if (savedWorkingDir) {
      return this.contextPool.getContext(savedWorkingDir);
    }

    return this.activeContext;
  }

  setActiveContext(nextActiveContext) {
    if (nextActiveContext !== this.activeContext) {
      if (this.activeContext === this.guessedContext) {
        this.guessedContext.destroy();
        this.guessedContext = null;
      }
      this.activeContext = nextActiveContext;
      this.rerender(() => {
        this.switchboard.didFinishContextChangeRender();
        this.switchboard.didFinishActiveContextUpdate();
      });
    } else {
      this.switchboard.didFinishActiveContextUpdate();
    }
  }

  async updateActiveContext(savedState = {}) {
    if (this.workspace.isDestroyed()) {
      return;
    }

    this.switchboard.didBeginActiveContextUpdate();

    const nextActiveContext = await this.getNextContext(savedState);
    this.setActiveContext(nextActiveContext);
  }

  refreshAtomGitRepository(workdir) {
    const atomGitRepo = this.project.getRepositories().find(repo => {
      return repo && _path2.default.normalize(repo.getWorkingDirectory()) === workdir;
    });
    return atomGitRepo ? atomGitRepo.refreshStatus() : Promise.resolve();
  }
}

exports.default = GithubPackage;
function pathForPaneItem(paneItem) {
  if (!paneItem) {
    return null;
  }

  // Likely GitHub package provided pane item
  if (typeof paneItem.getWorkingDirectory === 'function') {
    return paneItem.getWorkingDirectory();
  }

  // TextEditor-like
  if (typeof paneItem.getPath === 'function') {
    return paneItem.getPath();
  }

  // Oh well
  return null;
}
//# sourceMappingURL=data:application/json;charset=utf-8;base64,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