'use strict';

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.LargeRepoError = exports.GitError = undefined;

var _extends = Object.assign || function (target) { for (var i = 1; i < arguments.length; i++) { var source = arguments[i]; for (var key in source) { if (Object.prototype.hasOwnProperty.call(source, key)) { target[key] = source[key]; } } } return target; };

var _path = require('path');

var _path2 = _interopRequireDefault(_path);

var _os = require('os');

var _os2 = _interopRequireDefault(_os);

var _child_process = require('child_process');

var _child_process2 = _interopRequireDefault(_child_process);

var _fsExtra = require('fs-extra');

var _fsExtra2 = _interopRequireDefault(_fsExtra);

var _util = require('util');

var _util2 = _interopRequireDefault(_util);

var _electron = require('electron');

var _eventKit = require('event-kit');

var _dugite = require('dugite');

var _whatTheDiff = require('what-the-diff');

var _whatTheStatus = require('what-the-status');

var _gitPromptServer = require('./git-prompt-server');

var _gitPromptServer2 = _interopRequireDefault(_gitPromptServer);

var _gitTempDir = require('./git-temp-dir');

var _gitTempDir2 = _interopRequireDefault(_gitTempDir);

var _asyncQueue = require('./async-queue');

var _asyncQueue2 = _interopRequireDefault(_asyncQueue);

var _reporterProxy = require('./reporter-proxy');

var _helpers = require('./helpers');

var _gitTimingsView = require('./views/git-timings-view');

var _gitTimingsView2 = _interopRequireDefault(_gitTimingsView);

var _file = require('./models/patch/file');

var _file2 = _interopRequireDefault(_file);

var _workerManager = require('./worker-manager');

var _workerManager2 = _interopRequireDefault(_workerManager);

function _interopRequireDefault(obj) { return obj && obj.__esModule ? obj : { default: obj }; }

const MAX_STATUS_OUTPUT_LENGTH = 1024 * 1024 * 10;

let headless = null;
let execPathPromise = null;

class GitError extends Error {
  constructor(message) {
    super(message);
    this.message = message;
    this.stack = new Error().stack;
  }
}

exports.GitError = GitError;
class LargeRepoError extends Error {
  constructor(message) {
    super(message);
    this.message = message;
    this.stack = new Error().stack;
  }
}

exports.LargeRepoError = LargeRepoError; // ignored for the purposes of usage metrics tracking because they're noisy

const IGNORED_GIT_COMMANDS = ['cat-file', 'config', 'diff', 'for-each-ref', 'log', 'rev-parse', 'status'];

const DISABLE_COLOR_FLAGS = ['branch', 'diff', 'showBranch', 'status', 'ui'].reduce((acc, type) => {
  acc.unshift('-c', `color.${type}=false`);
  return acc;
}, []);

/**
 * Expand config path name per
 * https://git-scm.com/docs/git-config#git-config-pathname
 * this regex attempts to get the specified user's home directory
 * Ex: on Mac ~kuychaco/ is expanded to the specified user’s home directory (/Users/kuychaco)
 * Regex translation:
 * ^~ line starts with tilde
 * ([^/]*)/ captures non-forwardslash characters before first slash
 */
const EXPAND_TILDE_REGEX = new RegExp('^~([^/]*)/');

class GitShellOutStrategy {

  constructor(workingDir, options = {}) {
    this.workingDir = workingDir;
    if (options.queue) {
      this.commandQueue = options.queue;
    } else {
      const parallelism = options.parallelism || Math.max(3, _os2.default.cpus().length);
      this.commandQueue = new _asyncQueue2.default({ parallelism });
    }

    this.prompt = options.prompt || (query => Promise.reject());
    this.workerManager = options.workerManager;

    if (headless === null) {
      headless = !_electron.remote.getCurrentWindow().isVisible();
    }
  }

  /*
   * Provide an asynchronous callback to be used to request input from the user for git operations.
   *
   * `prompt` must be a callable that accepts a query object `{prompt, includeUsername}` and returns a Promise
   * that either resolves with a result object `{[username], password}` or rejects on cancellation.
   */
  setPromptCallback(prompt) {
    this.prompt = prompt;
  }

  // Execute a command and read the output using the embedded Git environment
  async exec(args, options = GitShellOutStrategy.defaultExecArgs) {
    /* eslint-disable no-console,no-control-regex */
    const { stdin, useGitPromptServer, useGpgWrapper, useGpgAtomPrompt, writeOperation } = options;
    const commandName = args[0];
    const subscriptions = new _eventKit.CompositeDisposable();
    const diagnosticsEnabled = process.env.ATOM_GITHUB_GIT_DIAGNOSTICS || atom.config.get('github.gitDiagnostics');

    const formattedArgs = `git ${args.join(' ')} in ${this.workingDir}`;
    const timingMarker = _gitTimingsView2.default.generateMarker(`git ${args.join(' ')}`);
    timingMarker.mark('queued');

    args.unshift(...DISABLE_COLOR_FLAGS);

    if (execPathPromise === null) {
      // Attempt to collect the --exec-path from a native git installation.
      execPathPromise = new Promise((resolve, reject) => {
        _child_process2.default.exec('git --exec-path', (error, stdout, stderr) => {
          if (error) {
            // Oh well
            resolve(null);
            return;
          }

          resolve(stdout.trim());
        });
      });
    }
    const execPath = await execPathPromise;

    return this.commandQueue.push(async () => {
      timingMarker.mark('prepare');
      let gitPromptServer;

      const pathParts = [];
      if (process.env.PATH) {
        pathParts.push(process.env.PATH);
      }
      if (execPath) {
        pathParts.push(execPath);
      }

      const env = _extends({}, process.env, {
        GIT_TERMINAL_PROMPT: '0',
        GIT_OPTIONAL_LOCKS: '0',
        PATH: pathParts.join(_path2.default.delimiter)
      });

      const gitTempDir = new _gitTempDir2.default();

      if (useGpgWrapper) {
        await gitTempDir.ensure();
        args.unshift('-c', `gpg.program=${gitTempDir.getGpgWrapperSh()}`);
      }

      if (useGitPromptServer) {
        gitPromptServer = new _gitPromptServer2.default(gitTempDir);
        await gitPromptServer.start(this.prompt);

        env.ATOM_GITHUB_TMP = gitTempDir.getRootPath();
        env.ATOM_GITHUB_ASKPASS_PATH = (0, _helpers.normalizeGitHelperPath)(gitTempDir.getAskPassJs());
        env.ATOM_GITHUB_CREDENTIAL_PATH = (0, _helpers.normalizeGitHelperPath)(gitTempDir.getCredentialHelperJs());
        env.ATOM_GITHUB_ELECTRON_PATH = (0, _helpers.normalizeGitHelperPath)((0, _helpers.getAtomHelperPath)());
        env.ATOM_GITHUB_SOCK_PATH = (0, _helpers.normalizeGitHelperPath)(gitTempDir.getSocketPath());

        env.ATOM_GITHUB_WORKDIR_PATH = this.workingDir;
        env.ATOM_GITHUB_DUGITE_PATH = (0, _helpers.getDugitePath)();
        env.ATOM_GITHUB_KEYTAR_STRATEGY_PATH = (0, _helpers.getSharedModulePath)('keytar-strategy');

        // "ssh" won't respect SSH_ASKPASS unless:
        // (a) it's running without a tty
        // (b) DISPLAY is set to something nonempty
        // But, on a Mac, DISPLAY is unset. Ensure that it is so our SSH_ASKPASS is respected.
        if (!process.env.DISPLAY || process.env.DISPLAY.length === 0) {
          env.DISPLAY = 'atom-github-placeholder';
        }

        env.ATOM_GITHUB_ORIGINAL_PATH = process.env.PATH || '';
        env.ATOM_GITHUB_ORIGINAL_GIT_ASKPASS = process.env.GIT_ASKPASS || '';
        env.ATOM_GITHUB_ORIGINAL_SSH_ASKPASS = process.env.SSH_ASKPASS || '';
        env.ATOM_GITHUB_ORIGINAL_GIT_SSH_COMMAND = process.env.GIT_SSH_COMMAND || '';
        env.ATOM_GITHUB_SPEC_MODE = atom.inSpecMode() ? 'true' : 'false';

        env.SSH_ASKPASS = (0, _helpers.normalizeGitHelperPath)(gitTempDir.getAskPassSh());
        env.GIT_ASKPASS = (0, _helpers.normalizeGitHelperPath)(gitTempDir.getAskPassSh());

        if (process.platform === 'linux') {
          env.GIT_SSH_COMMAND = gitTempDir.getSshWrapperSh();
        } else {
          env.GIT_SSH_COMMAND = process.env.GIT_SSH_COMMAND;
        }

        const credentialHelperSh = (0, _helpers.normalizeGitHelperPath)(gitTempDir.getCredentialHelperSh());
        args.unshift('-c', `credential.helper=${credentialHelperSh}`);
      }

      if (useGpgWrapper && useGitPromptServer && useGpgAtomPrompt) {
        env.ATOM_GITHUB_GPG_PROMPT = 'true';
      }

      if (diagnosticsEnabled) {
        env.GIT_TRACE = 'true';
        env.GIT_TRACE_CURL = 'true';
      }

      let opts = { env };

      if (stdin) {
        opts.stdin = stdin;
        opts.stdinEncoding = 'utf8';
      }

      if (process.env.PRINT_GIT_TIMES) {
        console.time(`git:${formattedArgs}`);
      }
      return new Promise(async (resolve, reject) => {
        if (options.beforeRun) {
          const newArgsOpts = await options.beforeRun({ args, opts });
          args = newArgsOpts.args;
          opts = newArgsOpts.opts;
        }
        const { promise, cancel } = this.executeGitCommand(args, opts, timingMarker);
        let expectCancel = false;
        if (gitPromptServer) {
          subscriptions.add(gitPromptServer.onDidCancel(async ({ handlerPid }) => {
            expectCancel = true;
            await cancel();

            // On Windows, the SSH_ASKPASS handler is executed as a non-child process, so the bin\git-askpass-atom.sh
            // process does not terminate when the git process is killed.
            // Kill the handler process *after* the git process has been killed to ensure that git doesn't have a
            // chance to fall back to GIT_ASKPASS from the credential handler.
            await new Promise((resolveKill, rejectKill) => {
              require('tree-kill')(handlerPid, 'SIGTERM', err => {
                if (err) {
                  rejectKill(err);
                } else {
                  resolveKill();
                }
              });
            });
          }));
        }

        const { stdout, stderr, exitCode, signal, timing } = await promise.catch(err => {
          if (err.signal) {
            return { signal: err.signal };
          }
          reject(err);
          return {};
        });

        if (timing) {
          const { execTime, spawnTime, ipcTime } = timing;
          const now = performance.now();
          timingMarker.mark('nexttick', now - execTime - spawnTime - ipcTime);
          timingMarker.mark('execute', now - execTime - ipcTime);
          timingMarker.mark('ipc', now - ipcTime);
        }
        timingMarker.finalize();
        if (process.env.PRINT_GIT_TIMES) {
          console.timeEnd(`git:${formattedArgs}`);
        }
        if (gitPromptServer) {
          gitPromptServer.terminate();
        }
        subscriptions.dispose();

        if (diagnosticsEnabled) {
          const exposeControlCharacters = raw => {
            if (!raw) {
              return '';
            }

            return raw.replace(/\u0000/ug, '<NUL>\n').replace(/\u001F/ug, '<SEP>');
          };

          if (headless) {
            let summary = `git:${formattedArgs}\n`;
            if (exitCode !== undefined) {
              summary += `exit status: ${exitCode}\n`;
            } else if (signal) {
              summary += `exit signal: ${signal}\n`;
            }
            if (stdin && stdin.length !== 0) {
              summary += `stdin:\n${exposeControlCharacters(stdin)}\n`;
            }
            summary += 'stdout:';
            if (stdout.length === 0) {
              summary += ' <empty>\n';
            } else {
              summary += `\n${exposeControlCharacters(stdout)}\n`;
            }
            summary += 'stderr:';
            if (stderr.length === 0) {
              summary += ' <empty>\n';
            } else {
              summary += `\n${exposeControlCharacters(stderr)}\n`;
            }

            console.log(summary);
          } else {
            const headerStyle = 'font-weight: bold; color: blue;';

            console.groupCollapsed(`git:${formattedArgs}`);
            if (exitCode !== undefined) {
              console.log('%cexit status%c %d', headerStyle, 'font-weight: normal; color: black;', exitCode);
            } else if (signal) {
              console.log('%cexit signal%c %s', headerStyle, 'font-weight: normal; color: black;', signal);
            }
            console.log('%cfull arguments%c %s', headerStyle, 'font-weight: normal; color: black;', _util2.default.inspect(args, { breakLength: Infinity }));
            if (stdin && stdin.length !== 0) {
              console.log('%cstdin', headerStyle);
              console.log(exposeControlCharacters(stdin));
            }
            console.log('%cstdout', headerStyle);
            console.log(exposeControlCharacters(stdout));
            console.log('%cstderr', headerStyle);
            console.log(exposeControlCharacters(stderr));
            console.groupEnd();
          }
        }

        if (exitCode !== 0 && !expectCancel) {
          const err = new GitError(`${formattedArgs} exited with code ${exitCode}\nstdout: ${stdout}\nstderr: ${stderr}`);
          err.code = exitCode;
          err.stdErr = stderr;
          err.stdOut = stdout;
          err.command = formattedArgs;
          reject(err);
        }

        if (!IGNORED_GIT_COMMANDS.includes(commandName)) {
          (0, _reporterProxy.incrementCounter)(commandName);
        }
        resolve(stdout);
      });
    }, { parallel: !writeOperation });
    /* eslint-enable no-console,no-control-regex */
  }

  async gpgExec(args, options) {
    try {
      return await this.exec(args.slice(), _extends({
        useGpgWrapper: true,
        useGpgAtomPrompt: false
      }, options));
    } catch (e) {
      if (/gpg failed/.test(e.stdErr)) {
        return await this.exec(args, _extends({
          useGitPromptServer: true,
          useGpgWrapper: true,
          useGpgAtomPrompt: true
        }, options));
      } else {
        throw e;
      }
    }
  }

  executeGitCommand(args, options, marker = null) {
    if (process.env.ATOM_GITHUB_INLINE_GIT_EXEC || !_workerManager2.default.getInstance().isReady()) {
      marker && marker.mark('nexttick');

      let childPid;
      options.processCallback = child => {
        childPid = child.pid;

        child.stdin.on('error', err => {
          throw new Error(`Error writing to stdin: git ${args.join(' ')} in ${this.workingDir}\n${options.stdin}\n${err}`);
        });
      };

      const promise = _dugite.GitProcess.exec(args, this.workingDir, options);
      marker && marker.mark('execute');
      return {
        promise,
        cancel: () => {
          if (!childPid) {
            return Promise.resolve();
          }

          return new Promise((resolve, reject) => {
            require('tree-kill')(childPid, 'SIGTERM', err => {
              if (err) {
                reject(err);
              } else {
                resolve();
              }
            });
          });
        }
      };
    } else {
      const workerManager = this.workerManager || _workerManager2.default.getInstance();
      return workerManager.request({
        args,
        workingDir: this.workingDir,
        options
      });
    }
  }

  async resolveDotGitDir() {
    try {
      await _fsExtra2.default.stat(this.workingDir); // fails if folder doesn't exist
      const output = await this.exec(['rev-parse', '--resolve-git-dir', _path2.default.join(this.workingDir, '.git')]);
      const dotGitDir = output.trim();
      if (_path2.default.isAbsolute(dotGitDir)) {
        return (0, _helpers.toNativePathSep)(dotGitDir);
      } else {
        return (0, _helpers.toNativePathSep)(_path2.default.resolve(_path2.default.join(this.workingDir, dotGitDir)));
      }
    } catch (e) {
      return null;
    }
  }

  init() {
    return this.exec(['init', this.workingDir]);
  }

  /**
   * Staging/Unstaging files and patches and committing
   */
  stageFiles(paths) {
    if (paths.length === 0) {
      return Promise.resolve(null);
    }
    const args = ['add'].concat(paths.map(_helpers.toGitPathSep));
    return this.exec(args, { writeOperation: true });
  }

  async fetchCommitMessageTemplate() {
    let templatePath = await this.getConfig('commit.template');
    if (!templatePath) {
      return null;
    }

    const homeDir = _os2.default.homedir();

    templatePath = templatePath.trim().replace(EXPAND_TILDE_REGEX, (_, user) => {
      // if no user is specified, fall back to using the home directory.
      return `${user ? _path2.default.join(_path2.default.dirname(homeDir), user) : homeDir}/`;
    });

    if (!_path2.default.isAbsolute(templatePath)) {
      templatePath = _path2.default.join(this.workingDir, templatePath);
    }

    if (!(await (0, _helpers.fileExists)(templatePath))) {
      throw new Error(`Invalid commit template path set in Git config: ${templatePath}`);
    }
    return await _fsExtra2.default.readFile(templatePath, { encoding: 'utf8' });
  }

  unstageFiles(paths, commit = 'HEAD') {
    if (paths.length === 0) {
      return Promise.resolve(null);
    }
    const args = ['reset', commit, '--'].concat(paths.map(_helpers.toGitPathSep));
    return this.exec(args, { writeOperation: true });
  }

  stageFileModeChange(filename, newMode) {
    const indexReadPromise = this.exec(['ls-files', '-s', '--', filename]);
    return this.exec(['update-index', '--cacheinfo', `${newMode},<OID_TBD>,${filename}`], {
      writeOperation: true,
      beforeRun: async function determineArgs({ args, opts }) {
        const index = await indexReadPromise;
        const oid = index.substr(7, 40);
        return {
          opts,
          args: ['update-index', '--cacheinfo', `${newMode},${oid},${filename}`]
        };
      }
    });
  }

  stageFileSymlinkChange(filename) {
    return this.exec(['rm', '--cached', filename], { writeOperation: true });
  }

  applyPatch(patch, { index } = {}) {
    const args = ['apply', '-'];
    if (index) {
      args.splice(1, 0, '--cached');
    }
    return this.exec(args, { stdin: patch, writeOperation: true });
  }

  async commit(rawMessage, { allowEmpty, amend, coAuthors, verbatim } = {}) {
    const args = ['commit'];
    let msg;

    // if amending and no new message is passed, use last commit's message. Ensure that we don't
    // mangle it in the process.
    if (amend && rawMessage.length === 0) {
      const { unbornRef, messageBody, messageSubject } = await this.getHeadCommit();
      if (unbornRef) {
        msg = rawMessage;
      } else {
        msg = `${messageSubject}\n\n${messageBody}`.trim();
        verbatim = true;
      }
    } else {
      msg = rawMessage;
    }

    // Determine the cleanup mode.
    if (verbatim) {
      args.push('--cleanup=verbatim');
    } else {
      const configured = await this.getConfig('commit.cleanup');
      const mode = configured && configured !== 'default' ? configured : 'strip';
      args.push(`--cleanup=${mode}`);
    }

    // add co-author commit trailers if necessary
    if (coAuthors && coAuthors.length > 0) {
      msg = await this.addCoAuthorsToMessage(msg, coAuthors);
    }

    args.push('-m', msg.trim());

    if (amend) {
      args.push('--amend');
    }
    if (allowEmpty) {
      args.push('--allow-empty');
    }
    return this.gpgExec(args, { writeOperation: true });
  }

  addCoAuthorsToMessage(message, coAuthors = []) {
    const trailers = coAuthors.map(author => {
      return {
        token: 'Co-Authored-By',
        value: `${author.name} <${author.email}>`
      };
    });

    // Ensure that message ends with newline for git-interpret trailers to work
    const msg = `${message.trim()}\n`;

    return trailers.length ? this.mergeTrailers(msg, trailers) : msg;
  }

  /**
   * File Status and Diffs
   */
  async getStatusBundle() {
    const args = ['status', '--porcelain=v2', '--branch', '--untracked-files=all', '--ignore-submodules=dirty', '-z'];
    const output = await this.exec(args);
    if (output.length > MAX_STATUS_OUTPUT_LENGTH) {
      throw new LargeRepoError();
    }

    const results = await (0, _whatTheStatus.parse)(output);

    for (const entryType in results) {
      if (Array.isArray(results[entryType])) {
        this.updateNativePathSepForEntries(results[entryType]);
      }
    }

    return results;
  }

  updateNativePathSepForEntries(entries) {
    entries.forEach(entry => {
      // Normally we would avoid mutating responses from other package's APIs, but we control
      // the `what-the-status` module and know there are no side effects.
      // This is a hot code path and by mutating we avoid creating new objects that will just be GC'ed
      if (entry.filePath) {
        entry.filePath = (0, _helpers.toNativePathSep)(entry.filePath);
      }
      if (entry.origFilePath) {
        entry.origFilePath = (0, _helpers.toNativePathSep)(entry.origFilePath);
      }
    });
  }

  async diffFileStatus(options = {}) {
    const args = ['diff', '--name-status', '--no-renames'];
    if (options.staged) {
      args.push('--staged');
    }
    if (options.target) {
      args.push(options.target);
    }
    const output = await this.exec(args);

    const statusMap = {
      A: 'added',
      M: 'modified',
      D: 'deleted',
      U: 'unmerged'
    };

    const fileStatuses = {};
    output && output.trim().split(_helpers.LINE_ENDING_REGEX).forEach(line => {
      const [status, rawFilePath] = line.split('\t');
      const filePath = (0, _helpers.toNativePathSep)(rawFilePath);
      fileStatuses[filePath] = statusMap[status];
    });
    if (!options.staged) {
      const untracked = await this.getUntrackedFiles();
      untracked.forEach(filePath => {
        fileStatuses[filePath] = 'added';
      });
    }
    return fileStatuses;
  }

  async getUntrackedFiles() {
    const output = await this.exec(['ls-files', '--others', '--exclude-standard']);
    if (output.trim() === '') {
      return [];
    }
    return output.trim().split(_helpers.LINE_ENDING_REGEX).map(_helpers.toNativePathSep);
  }

  async getDiffsForFilePath(filePath, { staged, baseCommit } = {}) {
    let args = ['diff', '--no-prefix', '--no-ext-diff', '--no-renames', '--diff-filter=u'];
    if (staged) {
      args.push('--staged');
    }
    if (baseCommit) {
      args.push(baseCommit);
    }
    args = args.concat(['--', (0, _helpers.toGitPathSep)(filePath)]);
    const output = await this.exec(args);

    let rawDiffs = [];
    if (output) {
      rawDiffs = (0, _whatTheDiff.parse)(output).filter(rawDiff => rawDiff.status !== 'unmerged');

      for (let i = 0; i < rawDiffs.length; i++) {
        const rawDiff = rawDiffs[i];
        if (rawDiff.oldPath) {
          rawDiff.oldPath = (0, _helpers.toNativePathSep)(rawDiff.oldPath);
        }
        if (rawDiff.newPath) {
          rawDiff.newPath = (0, _helpers.toNativePathSep)(rawDiff.newPath);
        }
      }
    }

    if (!staged && (await this.getUntrackedFiles()).includes(filePath)) {
      // add untracked file
      const absPath = _path2.default.join(this.workingDir, filePath);
      const executable = await (0, _helpers.isFileExecutable)(absPath);
      const symlink = await (0, _helpers.isFileSymlink)(absPath);
      const contents = await _fsExtra2.default.readFile(absPath, { encoding: 'utf8' });
      const binary = (0, _helpers.isBinary)(contents);
      let mode;
      let realpath;
      if (executable) {
        mode = _file2.default.modes.EXECUTABLE;
      } else if (symlink) {
        mode = _file2.default.modes.SYMLINK;
        realpath = await _fsExtra2.default.realpath(absPath);
      } else {
        mode = _file2.default.modes.NORMAL;
      }

      rawDiffs.push(buildAddedFilePatch(filePath, binary ? null : contents, mode, realpath));
    }
    if (rawDiffs.length > 2) {
      throw new Error(`Expected between 0 and 2 diffs for ${filePath} but got ${rawDiffs.length}`);
    }
    return rawDiffs;
  }

  async getStagedChangesPatch() {
    const output = await this.exec(['diff', '--staged', '--no-prefix', '--no-ext-diff', '--no-renames', '--diff-filter=u']);

    if (!output) {
      return [];
    }

    const diffs = (0, _whatTheDiff.parse)(output);
    for (const diff of diffs) {
      if (diff.oldPath) {
        diff.oldPath = (0, _helpers.toNativePathSep)(diff.oldPath);
      }
      if (diff.newPath) {
        diff.newPath = (0, _helpers.toNativePathSep)(diff.newPath);
      }
    }
    return diffs;
  }

  /**
   * Miscellaneous getters
   */
  async getCommit(ref) {
    const [commit] = await this.getCommits({ max: 1, ref, includeUnborn: true });
    return commit;
  }

  async getHeadCommit() {
    const [headCommit] = await this.getCommits({ max: 1, ref: 'HEAD', includeUnborn: true });
    return headCommit;
  }

  async getCommits(options = {}) {
    const { max, ref, includeUnborn } = _extends({ max: 1, ref: 'HEAD', includeUnborn: false }, options);

    // https://git-scm.com/docs/git-log#_pretty_formats
    // %x00 - null byte
    // %H - commit SHA
    // %ae - author email
    // %at - timestamp, UNIX timestamp
    // %s - subject
    // %b - body
    const output = await this.exec(['log', '--pretty=format:%H%x00%ae%x00%at%x00%s%x00%b', '--no-abbrev-commit', '-z', '-n', max, ref, '--']).catch(err => {
      if (/unknown revision/.test(err.stdErr) || /bad revision 'HEAD'/.test(err.stdErr)) {
        return '';
      } else {
        throw err;
      }
    });

    if (output === '') {
      return includeUnborn ? [{ sha: '', message: '', unbornRef: true }] : [];
    }

    const fields = output.trim().split('\0');
    const commits = [];
    for (let i = 0; i < fields.length; i += 5) {
      const body = fields[i + 4];

      const { message: messageBody, coAuthors } = (0, _helpers.extractCoAuthorsAndRawCommitMessage)(body);

      commits.push({
        sha: fields[i] && fields[i].trim(),
        authorEmail: fields[i + 1] && fields[i + 1].trim(),
        authorDate: parseInt(fields[i + 2], 10),
        messageSubject: fields[i + 3],
        messageBody,
        coAuthors,
        unbornRef: false
      });
    }
    return commits;
  }

  async getAuthors(options = {}) {
    const { max, ref } = _extends({ max: 1, ref: 'HEAD' }, options);

    // https://git-scm.com/docs/git-log#_pretty_formats
    // %x1F - field separator byte
    // %an - author name
    // %ae - author email
    // %cn - committer name
    // %ce - committer email
    // %(trailers:unfold,only) - the commit message trailers, separated
    //                           by newlines and unfolded (i.e. properly
    //                           formatted and one trailer per line).

    const delimiter = '1F';
    const delimiterString = String.fromCharCode(parseInt(delimiter, 16));
    const fields = ['%an', '%ae', '%cn', '%ce', '%(trailers:unfold,only)'];
    const format = fields.join(`%x${delimiter}`);

    try {
      const output = await this.exec(['log', `--format=${format}`, '-z', '-n', max, ref, '--']);

      return output.split('\0').reduce((acc, line) => {
        if (line.length === 0) {
          return acc;
        }

        const [an, ae, cn, ce, trailers] = line.split(delimiterString);
        trailers.split('\n').map(trailer => trailer.match(_helpers.CO_AUTHOR_REGEX)).filter(match => match !== null).forEach(([_, name, email]) => {
          acc[email] = name;
        });

        acc[ae] = an;
        acc[ce] = cn;

        return acc;
      }, {});
    } catch (err) {
      if (/unknown revision/.test(err.stdErr) || /bad revision 'HEAD'/.test(err.stdErr)) {
        return [];
      } else {
        throw err;
      }
    }
  }

  mergeTrailers(commitMessage, trailers, unfold) {
    const args = ['interpret-trailers'];
    if (unfold) {
      args.push('--unfold');
    }
    for (const trailer of trailers) {
      args.push('--trailer', `${trailer.token}=${trailer.value}`);
    }
    return this.exec(args, { stdin: commitMessage });
  }

  readFileFromIndex(filePath) {
    return this.exec(['show', `:${(0, _helpers.toGitPathSep)(filePath)}`]);
  }

  /**
   * Merge
   */
  merge(branchName) {
    return this.gpgExec(['merge', branchName], { writeOperation: true });
  }

  isMerging(dotGitDir) {
    return (0, _helpers.fileExists)(_path2.default.join(dotGitDir, 'MERGE_HEAD')).catch(() => false);
  }

  abortMerge() {
    return this.exec(['merge', '--abort'], { writeOperation: true });
  }

  checkoutSide(side, paths) {
    if (paths.length === 0) {
      return Promise.resolve();
    }

    return this.exec(['checkout', `--${side}`, ...paths.map(_helpers.toGitPathSep)]);
  }

  /**
   * Rebase
   */
  async isRebasing(dotGitDir) {
    const results = await Promise.all([(0, _helpers.fileExists)(_path2.default.join(dotGitDir, 'rebase-merge')), (0, _helpers.fileExists)(_path2.default.join(dotGitDir, 'rebase-apply'))]);
    return results.some(r => r);
  }

  /**
   * Remote interactions
   */
  clone(remoteUrl, options = {}) {
    const args = ['clone'];
    if (options.noLocal) {
      args.push('--no-local');
    }
    if (options.bare) {
      args.push('--bare');
    }
    if (options.recursive) {
      args.push('--recursive');
    }
    args.push(remoteUrl, this.workingDir);

    return this.exec(args, { useGitPromptServer: true, writeOperation: true });
  }

  fetch(remoteName, branchName) {
    return this.exec(['fetch', remoteName, branchName], { useGitPromptServer: true, writeOperation: true });
  }

  pull(remoteName, branchName, options = {}) {
    const args = ['pull', remoteName, options.refSpec || branchName];
    if (options.ffOnly) {
      args.push('--ff-only');
    }
    return this.gpgExec(args, { useGitPromptServer: true, writeOperation: true });
  }

  push(remoteName, branchName, options = {}) {
    const args = ['push', remoteName || 'origin', options.refSpec || `refs/heads/${branchName}`];
    if (options.setUpstream) {
      args.push('--set-upstream');
    }
    if (options.force) {
      args.push('--force');
    }
    return this.exec(args, { useGitPromptServer: true, writeOperation: true });
  }

  /**
   * Undo Operations
   */
  reset(type, revision = 'HEAD') {
    const validTypes = ['soft'];
    if (!validTypes.includes(type)) {
      throw new Error(`Invalid type ${type}. Must be one of: ${validTypes.join(', ')}`);
    }
    return this.exec(['reset', `--${type}`, revision]);
  }

  deleteRef(ref) {
    return this.exec(['update-ref', '-d', ref]);
  }

  /**
   * Branches
   */
  checkout(branchName, options = {}) {
    const args = ['checkout'];
    if (options.createNew) {
      args.push('-b');
    }
    args.push(branchName);
    if (options.startPoint) {
      if (options.track) {
        args.push('--track');
      }
      args.push(options.startPoint);
    }

    return this.exec(args, { writeOperation: true });
  }

  async getBranches() {
    const format = ['%(objectname)', '%(HEAD)', '%(refname:short)', '%(upstream)', '%(upstream:remotename)', '%(upstream:remoteref)', '%(push)', '%(push:remotename)', '%(push:remoteref)'].join('%00');

    const output = await this.exec(['for-each-ref', `--format=${format}`, 'refs/heads/**']);
    return output.trim().split(_helpers.LINE_ENDING_REGEX).map(line => {
      const [sha, head, name, upstreamTrackingRef, upstreamRemoteName, upstreamRemoteRef, pushTrackingRef, pushRemoteName, pushRemoteRef] = line.split('\0');

      const branch = { name, sha, head: head === '*' };
      if (upstreamTrackingRef || upstreamRemoteName || upstreamRemoteRef) {
        branch.upstream = {
          trackingRef: upstreamTrackingRef,
          remoteName: upstreamRemoteName,
          remoteRef: upstreamRemoteRef
        };
      }
      if (branch.upstream || pushTrackingRef || pushRemoteName || pushRemoteRef) {
        branch.push = {
          trackingRef: pushTrackingRef,
          remoteName: pushRemoteName || branch.upstream && branch.upstream.remoteName,
          remoteRef: pushRemoteRef || branch.upstream && branch.upstream.remoteRef
        };
      }
      return branch;
    });
  }

  checkoutFiles(paths, revision) {
    if (paths.length === 0) {
      return null;
    }
    const args = ['checkout'];
    if (revision) {
      args.push(revision);
    }
    return this.exec(args.concat('--', paths.map(_helpers.toGitPathSep)), { writeOperation: true });
  }

  async describeHead() {
    return (await this.exec(['describe', '--contains', '--all', '--always', 'HEAD'])).trim();
  }

  async getConfig(option, { local } = {}) {
    let output;
    try {
      let args = ['config'];
      if (local || atom.inSpecMode()) {
        args.push('--local');
      }
      args = args.concat(option);
      output = await this.exec(args);
    } catch (err) {
      if (err.code === 1) {
        // No matching config found
        return null;
      } else {
        throw err;
      }
    }

    return output.trim();
  }

  setConfig(option, value, { replaceAll } = {}) {
    let args = ['config'];
    if (replaceAll) {
      args.push('--replace-all');
    }
    args = args.concat(option, value);
    return this.exec(args, { writeOperation: true });
  }

  unsetConfig(option) {
    return this.exec(['config', '--unset', option], { writeOperation: true });
  }

  async getRemotes() {
    let output = await this.getConfig(['--get-regexp', '^remote\\..*\\.url$'], { local: true });
    if (output) {
      output = output.trim();
      if (!output.length) {
        return [];
      }
      return output.split('\n').map(line => {
        const match = line.match(/^remote\.(.*)\.url (.*)$/);
        return {
          name: match[1],
          url: match[2]
        };
      });
    } else {
      return [];
    }
  }

  addRemote(name, url) {
    return this.exec(['remote', 'add', name, url]);
  }

  async createBlob({ filePath, stdin } = {}) {
    let output;
    if (filePath) {
      try {
        output = (await this.exec(['hash-object', '-w', filePath], { writeOperation: true })).trim();
      } catch (e) {
        if (e.stdErr && e.stdErr.match(/fatal: Cannot open .*: No such file or directory/)) {
          output = null;
        } else {
          throw e;
        }
      }
    } else if (stdin) {
      output = (await this.exec(['hash-object', '-w', '--stdin'], { stdin, writeOperation: true })).trim();
    } else {
      throw new Error('Must supply file path or stdin');
    }
    return output;
  }

  async expandBlobToFile(absFilePath, sha) {
    const output = await this.exec(['cat-file', '-p', sha]);
    await _fsExtra2.default.writeFile(absFilePath, output, { encoding: 'utf8' });
    return absFilePath;
  }

  async getBlobContents(sha) {
    return await this.exec(['cat-file', '-p', sha]);
  }

  async mergeFile(oursPath, commonBasePath, theirsPath, resultPath) {
    const args = ['merge-file', '-p', oursPath, commonBasePath, theirsPath, '-L', 'current', '-L', 'after discard', '-L', 'before discard'];
    let output;
    let conflict = false;
    try {
      output = await this.exec(args);
    } catch (e) {
      if (e instanceof GitError && e.code === 1) {
        output = e.stdOut;
        conflict = true;
      } else {
        throw e;
      }
    }

    // Interpret a relative resultPath as relative to the repository working directory for consistency with the
    // other arguments.
    const resolvedResultPath = _path2.default.resolve(this.workingDir, resultPath);
    await _fsExtra2.default.writeFile(resolvedResultPath, output, { encoding: 'utf8' });

    return { filePath: oursPath, resultPath, conflict };
  }

  async writeMergeConflictToIndex(filePath, commonBaseSha, oursSha, theirsSha) {
    const gitFilePath = (0, _helpers.toGitPathSep)(filePath);
    const fileMode = await this.getFileMode(filePath);
    let indexInfo = `0 0000000000000000000000000000000000000000\t${gitFilePath}\n`;
    if (commonBaseSha) {
      indexInfo += `${fileMode} ${commonBaseSha} 1\t${gitFilePath}\n`;
    }
    if (oursSha) {
      indexInfo += `${fileMode} ${oursSha} 2\t${gitFilePath}\n`;
    }
    if (theirsSha) {
      indexInfo += `${fileMode} ${theirsSha} 3\t${gitFilePath}\n`;
    }
    return this.exec(['update-index', '--index-info'], { stdin: indexInfo, writeOperation: true });
  }

  async getFileMode(filePath) {
    const output = await this.exec(['ls-files', '--stage', '--', (0, _helpers.toGitPathSep)(filePath)]);
    if (output) {
      return output.slice(0, 6);
    } else {
      const executable = await (0, _helpers.isFileExecutable)(_path2.default.join(this.workingDir, filePath));
      const symlink = await (0, _helpers.isFileSymlink)(_path2.default.join(this.workingDir, filePath));
      if (symlink) {
        return _file2.default.modes.SYMLINK;
      } else if (executable) {
        return _file2.default.modes.EXECUTABLE;
      } else {
        return _file2.default.modes.NORMAL;
      }
    }
  }

  destroy() {
    this.commandQueue.dispose();
  }
}

exports.default = GitShellOutStrategy;
GitShellOutStrategy.defaultExecArgs = {
  stdin: null,
  useGitPromptServer: false,
  useGpgWrapper: false,
  useGpgAtomPrompt: false,
  writeOperation: false
};
function buildAddedFilePatch(filePath, contents, mode, realpath) {
  const hunks = [];
  if (contents) {
    let noNewLine;
    let lines;
    if (mode === _file2.default.modes.SYMLINK) {
      noNewLine = false;
      lines = [`+${(0, _helpers.toGitPathSep)(realpath)}`, '\\ No newline at end of file'];
    } else {
      noNewLine = contents[contents.length - 1] !== '\n';
      lines = contents.trim().split(_helpers.LINE_ENDING_REGEX).map(line => `+${line}`);
    }
    if (noNewLine) {
      lines.push('\\ No newline at end of file');
    }
    hunks.push({
      lines,
      oldStartLine: 0,
      oldLineCount: 0,
      newStartLine: 1,
      heading: '',
      newLineCount: noNewLine ? lines.length - 1 : lines.length
    });
  }
  return {
    oldPath: null,
    newPath: (0, _helpers.toNativePathSep)(filePath),
    oldMode: null,
    newMode: mode,
    status: 'added',
    hunks
  };
}
//# sourceMappingURL=data:application/json;charset=utf-8;base64,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