'use strict';

Object.defineProperty(exports, "__esModule", {
  value: true
});

var _extends = Object.assign || function (target) { for (var i = 1; i < arguments.length; i++) { var source = arguments[i]; for (var key in source) { if (Object.prototype.hasOwnProperty.call(source, key)) { target[key] = source[key]; } } } return target; };

var _fsExtra = require('fs-extra');

var _fsExtra2 = _interopRequireDefault(_fsExtra);

var _path = require('path');

var _path2 = _interopRequireDefault(_path);

var _electron = require('electron');

var _react = require('react');

var _react2 = _interopRequireDefault(_react);

var _propTypes = require('prop-types');

var _propTypes2 = _interopRequireDefault(_propTypes);

var _eventKit = require('event-kit');

var _statusBar = require('../atom/status-bar');

var _statusBar2 = _interopRequireDefault(_statusBar);

var _panel = require('../atom/panel');

var _panel2 = _interopRequireDefault(_panel);

var _paneItem = require('../atom/pane-item');

var _paneItem2 = _interopRequireDefault(_paneItem);

var _cloneDialog = require('../views/clone-dialog');

var _cloneDialog2 = _interopRequireDefault(_cloneDialog);

var _openIssueishDialog = require('../views/open-issueish-dialog');

var _openIssueishDialog2 = _interopRequireDefault(_openIssueishDialog);

var _initDialog = require('../views/init-dialog');

var _initDialog2 = _interopRequireDefault(_initDialog);

var _credentialDialog = require('../views/credential-dialog');

var _credentialDialog2 = _interopRequireDefault(_credentialDialog);

var _commands = require('../atom/commands');

var _commands2 = _interopRequireDefault(_commands);

var _gitTimingsView = require('../views/git-timings-view');

var _gitTimingsView2 = _interopRequireDefault(_gitTimingsView);

var _changedFileItem = require('../items/changed-file-item');

var _changedFileItem2 = _interopRequireDefault(_changedFileItem);

var _issueishDetailItem = require('../items/issueish-detail-item');

var _issueishDetailItem2 = _interopRequireDefault(_issueishDetailItem);

var _commitPreviewItem = require('../items/commit-preview-item');

var _commitPreviewItem2 = _interopRequireDefault(_commitPreviewItem);

var _gitTabItem = require('../items/git-tab-item');

var _gitTabItem2 = _interopRequireDefault(_gitTabItem);

var _githubTabItem = require('../items/github-tab-item');

var _githubTabItem2 = _interopRequireDefault(_githubTabItem);

var _statusBarTileController = require('./status-bar-tile-controller');

var _statusBarTileController2 = _interopRequireDefault(_statusBarTileController);

var _repositoryConflictController = require('./repository-conflict-controller');

var _repositoryConflictController2 = _interopRequireDefault(_repositoryConflictController);

var _gitCacheView = require('../views/git-cache-view');

var _gitCacheView2 = _interopRequireDefault(_gitCacheView);

var _conflict = require('../models/conflicts/conflict');

var _conflict2 = _interopRequireDefault(_conflict);

var _switchboard = require('../switchboard');

var _switchboard2 = _interopRequireDefault(_switchboard);

var _propTypes3 = require('../prop-types');

var _helpers = require('../helpers');

var _gitShellOutStrategy = require('../git-shell-out-strategy');

var _reporterProxy = require('../reporter-proxy');

function _interopRequireDefault(obj) { return obj && obj.__esModule ? obj : { default: obj }; }

class RootController extends _react2.default.Component {

  constructor(props, context) {
    super(props, context);

    this.toggleCommitPreviewItem = () => {
      const workdir = this.props.repository.getWorkingDirectoryPath();
      return this.props.workspace.toggle(_commitPreviewItem2.default.buildURI(workdir));
    };

    this.surfaceFromFileAtPath = (filePath, stagingStatus) => {
      const gitTab = this.gitTabTracker.getComponent();
      return gitTab && gitTab.focusAndSelectStagingItem(filePath, stagingStatus);
    };

    this.surfaceToCommitPreviewButton = () => {
      const gitTab = this.gitTabTracker.getComponent();
      return gitTab && gitTab.focusAndSelectCommitPreviewButton();
    };

    (0, _helpers.autobind)(this, 'installReactDevTools', 'clearGithubToken', 'initializeRepo', 'showOpenIssueishDialog', 'showWaterfallDiagnostics', 'showCacheDiagnostics', 'acceptClone', 'cancelClone', 'acceptInit', 'cancelInit', 'acceptOpenIssueish', 'cancelOpenIssueish', 'destroyFilePatchPaneItems', 'destroyEmptyFilePatchPaneItems', 'openCloneDialog', 'quietlySelectItem', 'viewUnstagedChangesForCurrentFile', 'viewStagedChangesForCurrentFile', 'openFiles', 'getUnsavedFiles', 'ensureNoUnsavedFiles', 'discardWorkDirChangesForPaths', 'discardLines', 'undoLastDiscard', 'refreshResolutionProgress');

    this.state = {
      cloneDialogActive: false,
      cloneDialogInProgress: false,
      initDialogActive: false,
      initDialogPath: null,
      initDialogResolve: null,
      credentialDialogQuery: null
    };

    this.gitTabTracker = new TabTracker('git', {
      uri: _gitTabItem2.default.buildURI(),
      getWorkspace: () => this.props.workspace
    });

    this.githubTabTracker = new TabTracker('github', {
      uri: _githubTabItem2.default.buildURI(),
      getWorkspace: () => this.props.workspace
    });

    this.subscription = new _eventKit.CompositeDisposable(this.props.repository.onMergeError(this.gitTabTracker.ensureVisible));

    this.props.commandRegistry.onDidDispatch(event => {
      if (event.type && event.type.startsWith('github:') && event.detail && event.detail[0] && event.detail[0].contextCommand) {
        (0, _reporterProxy.addEvent)('context-menu-action', {
          package: 'github',
          command: event.type
        });
      }
    });
  }

  componentDidMount() {
    this.openTabs();
  }

  render() {
    return _react2.default.createElement(
      _react.Fragment,
      null,
      this.renderCommands(),
      this.renderStatusBarTile(),
      this.renderPaneItems(),
      this.renderDialogs(),
      this.renderConflictResolver()
    );
  }

  renderCommands() {
    const devMode = global.atom && global.atom.inDevMode();

    return _react2.default.createElement(
      _commands2.default,
      { registry: this.props.commandRegistry, target: 'atom-workspace' },
      devMode && _react2.default.createElement(_commands.Command, { command: 'github:install-react-dev-tools', callback: this.installReactDevTools }),
      _react2.default.createElement(_commands.Command, { command: 'github:toggle-commit-preview', callback: this.toggleCommitPreviewItem }),
      _react2.default.createElement(_commands.Command, { command: 'github:logout', callback: this.clearGithubToken }),
      _react2.default.createElement(_commands.Command, { command: 'github:show-waterfall-diagnostics', callback: this.showWaterfallDiagnostics }),
      _react2.default.createElement(_commands.Command, { command: 'github:show-cache-diagnostics', callback: this.showCacheDiagnostics }),
      _react2.default.createElement(_commands.Command, { command: 'github:open-issue-or-pull-request', callback: this.showOpenIssueishDialog }),
      _react2.default.createElement(_commands.Command, { command: 'github:toggle-git-tab', callback: this.gitTabTracker.toggle }),
      _react2.default.createElement(_commands.Command, { command: 'github:toggle-git-tab-focus', callback: this.gitTabTracker.toggleFocus }),
      _react2.default.createElement(_commands.Command, { command: 'github:toggle-github-tab', callback: this.githubTabTracker.toggle }),
      _react2.default.createElement(_commands.Command, { command: 'github:toggle-github-tab-focus', callback: this.githubTabTracker.toggleFocus }),
      _react2.default.createElement(_commands.Command, { command: 'github:clone', callback: this.openCloneDialog }),
      _react2.default.createElement(_commands.Command, {
        command: 'github:view-unstaged-changes-for-current-file',
        callback: this.viewUnstagedChangesForCurrentFile
      }),
      _react2.default.createElement(_commands.Command, {
        command: 'github:view-staged-changes-for-current-file',
        callback: this.viewStagedChangesForCurrentFile
      }),
      _react2.default.createElement(_commands.Command, {
        command: 'github:close-all-diff-views',
        callback: this.destroyFilePatchPaneItems
      }),
      _react2.default.createElement(_commands.Command, {
        command: 'github:close-empty-diff-views',
        callback: this.destroyEmptyFilePatchPaneItems
      })
    );
  }

  renderStatusBarTile() {
    return _react2.default.createElement(
      _statusBar2.default,
      {
        statusBar: this.props.statusBar,
        onConsumeStatusBar: sb => this.onConsumeStatusBar(sb),
        className: 'github-StatusBarTileController' },
      _react2.default.createElement(_statusBarTileController2.default, {
        pipelineManager: this.props.pipelineManager,
        workspace: this.props.workspace,
        repository: this.props.repository,
        commandRegistry: this.props.commandRegistry,
        notificationManager: this.props.notificationManager,
        tooltips: this.props.tooltips,
        confirm: this.props.confirm,
        toggleGitTab: this.gitTabTracker.toggle,
        toggleGithubTab: this.githubTabTracker.toggle,
        ensureGitTabVisible: this.gitTabTracker.ensureVisible
      })
    );
  }

  renderDialogs() {
    return _react2.default.createElement(
      _react.Fragment,
      null,
      this.renderInitDialog(),
      this.renderCloneDialog(),
      this.renderCredentialDialog(),
      this.renderOpenIssueishDialog()
    );
  }

  renderInitDialog() {
    if (!this.state.initDialogActive) {
      return null;
    }

    return _react2.default.createElement(
      _panel2.default,
      { workspace: this.props.workspace, location: 'modal' },
      _react2.default.createElement(_initDialog2.default, {
        config: this.props.config,
        commandRegistry: this.props.commandRegistry,
        initPath: this.state.initDialogPath,
        didAccept: this.acceptInit,
        didCancel: this.cancelInit
      })
    );
  }

  renderCloneDialog() {
    if (!this.state.cloneDialogActive) {
      return null;
    }

    return _react2.default.createElement(
      _panel2.default,
      { workspace: this.props.workspace, location: 'modal' },
      _react2.default.createElement(_cloneDialog2.default, {
        config: this.props.config,
        commandRegistry: this.props.commandRegistry,
        didAccept: this.acceptClone,
        didCancel: this.cancelClone,
        inProgress: this.state.cloneDialogInProgress
      })
    );
  }

  renderOpenIssueishDialog() {
    if (!this.state.openIssueishDialogActive) {
      return null;
    }

    return _react2.default.createElement(
      _panel2.default,
      { workspace: this.props.workspace, location: 'modal' },
      _react2.default.createElement(_openIssueishDialog2.default, {
        commandRegistry: this.props.commandRegistry,
        didAccept: this.acceptOpenIssueish,
        didCancel: this.cancelOpenIssueish
      })
    );
  }

  renderCredentialDialog() {
    if (this.state.credentialDialogQuery === null) {
      return null;
    }

    return _react2.default.createElement(
      _panel2.default,
      { workspace: this.props.workspace, location: 'modal' },
      _react2.default.createElement(_credentialDialog2.default, _extends({ commandRegistry: this.props.commandRegistry }, this.state.credentialDialogQuery))
    );
  }

  renderConflictResolver() {
    if (!this.props.repository) {
      return null;
    }

    return _react2.default.createElement(_repositoryConflictController2.default, {
      workspace: this.props.workspace,
      config: this.props.config,
      repository: this.props.repository,
      resolutionProgress: this.props.resolutionProgress,
      refreshResolutionProgress: this.refreshResolutionProgress,
      commandRegistry: this.props.commandRegistry
    });
  }

  renderPaneItems() {
    return _react2.default.createElement(
      _react.Fragment,
      null,
      _react2.default.createElement(
        _paneItem2.default,
        {
          workspace: this.props.workspace,
          uriPattern: _gitTabItem2.default.uriPattern,
          className: 'github-Git-root' },
        ({ itemHolder }) => _react2.default.createElement(_gitTabItem2.default, {
          ref: itemHolder.setter,
          workspace: this.props.workspace,
          commandRegistry: this.props.commandRegistry,
          notificationManager: this.props.notificationManager,
          tooltips: this.props.tooltips,
          grammars: this.props.grammars,
          project: this.props.project,
          confirm: this.props.confirm,
          config: this.props.config,
          repository: this.props.repository,
          loginModel: this.props.loginModel,
          initializeRepo: this.initializeRepo,
          resolutionProgress: this.props.resolutionProgress,
          ensureGitTab: this.gitTabTracker.ensureVisible,
          openFiles: this.openFiles,
          discardWorkDirChangesForPaths: this.discardWorkDirChangesForPaths,
          undoLastDiscard: this.undoLastDiscard,
          refreshResolutionProgress: this.refreshResolutionProgress
        })
      ),
      _react2.default.createElement(
        _paneItem2.default,
        {
          workspace: this.props.workspace,
          uriPattern: _githubTabItem2.default.uriPattern,
          className: 'github-GitHub-root' },
        ({ itemHolder }) => _react2.default.createElement(_githubTabItem2.default, {
          ref: itemHolder.setter,
          repository: this.props.repository,
          loginModel: this.props.loginModel,
          workspace: this.props.workspace
        })
      ),
      _react2.default.createElement(
        _paneItem2.default,
        {
          workspace: this.props.workspace,
          uriPattern: _changedFileItem2.default.uriPattern },
        ({ itemHolder, params }) => _react2.default.createElement(_changedFileItem2.default, {
          ref: itemHolder.setter,

          workdirContextPool: this.props.workdirContextPool,
          relPath: _path2.default.join(...params.relPath),
          workingDirectory: params.workingDirectory,
          stagingStatus: params.stagingStatus,

          tooltips: this.props.tooltips,
          commands: this.props.commandRegistry,
          keymaps: this.props.keymaps,
          workspace: this.props.workspace,
          config: this.props.config,

          discardLines: this.discardLines,
          undoLastDiscard: this.undoLastDiscard,
          surfaceFileAtPath: this.surfaceFromFileAtPath
        })
      ),
      _react2.default.createElement(
        _paneItem2.default,
        {
          workspace: this.props.workspace,
          uriPattern: _commitPreviewItem2.default.uriPattern,
          className: 'github-CommitPreview-root' },
        ({ itemHolder, params }) => _react2.default.createElement(_commitPreviewItem2.default, {
          ref: itemHolder.setter,

          workdirContextPool: this.props.workdirContextPool,
          workingDirectory: params.workingDirectory,
          workspace: this.props.workspace,
          commands: this.props.commandRegistry,
          keymaps: this.props.keymaps,
          tooltips: this.props.tooltips,
          config: this.props.config,

          discardLines: this.discardLines,
          undoLastDiscard: this.undoLastDiscard,
          surfaceToCommitPreviewButton: this.surfaceToCommitPreviewButton
        })
      ),
      _react2.default.createElement(
        _paneItem2.default,
        { workspace: this.props.workspace, uriPattern: _issueishDetailItem2.default.uriPattern },
        ({ itemHolder, params }) => _react2.default.createElement(_issueishDetailItem2.default, {
          ref: itemHolder.setter,

          host: params.host,
          owner: params.owner,
          repo: params.repo,
          issueishNumber: parseInt(params.issueishNumber, 10),

          workingDirectory: params.workingDirectory,
          workdirContextPool: this.props.workdirContextPool,
          loginModel: this.props.loginModel
        })
      ),
      _react2.default.createElement(
        _paneItem2.default,
        { workspace: this.props.workspace, uriPattern: _gitTimingsView2.default.uriPattern },
        ({ itemHolder }) => _react2.default.createElement(_gitTimingsView2.default, { ref: itemHolder.setter })
      ),
      _react2.default.createElement(
        _paneItem2.default,
        { workspace: this.props.workspace, uriPattern: _gitCacheView2.default.uriPattern },
        ({ itemHolder }) => _react2.default.createElement(_gitCacheView2.default, { ref: itemHolder.setter, repository: this.props.repository })
      )
    );
  }

  async openTabs() {
    if (this.props.startOpen) {
      await Promise.all([this.gitTabTracker.ensureRendered(false), this.githubTabTracker.ensureRendered(false)]);
    }

    if (this.props.startRevealed) {
      const docks = new Set([_gitTabItem2.default.buildURI(), _githubTabItem2.default.buildURI()].map(uri => this.props.workspace.paneContainerForURI(uri)).filter(container => container && typeof container.show === 'function'));

      for (const dock of docks) {
        dock.show();
      }
    }
  }

  async installReactDevTools() {
    // Prevent electron-link from attempting to descend into electron-devtools-installer, which is not available
    // when we're bundled in Atom.
    const devToolsName = 'electron-devtools-installer';
    const devTools = require(devToolsName);

    const crossUnzipName = 'cross-unzip';
    const unzip = require(crossUnzipName);

    const reactId = devTools.REACT_DEVELOPER_TOOLS.id;

    const url = 'https://clients2.google.com/service/update2/crx?' + `response=redirect&x=id%3D${reactId}%26uc&prodversion=32`;
    const extensionFolder = _path2.default.resolve(_electron.remote.app.getPath('userData'), `extensions/${reactId}`);
    const extensionFile = `${extensionFolder}.crx`;
    await _fsExtra2.default.ensureDir(_path2.default.dirname(extensionFile));
    const response = await fetch(url, { method: 'GET' });
    const body = Buffer.from((await response.arrayBuffer()));
    await _fsExtra2.default.writeFile(extensionFile, body);

    await new Promise((resolve, reject) => {
      unzip(extensionFile, extensionFolder, async err => {
        if (err && !(await _fsExtra2.default.exists(_path2.default.join(extensionFolder, 'manifest.json')))) {
          reject(err);
        }

        resolve();
      });
    });

    await _fsExtra2.default.ensureDir(extensionFolder, 0o755);
    await devTools.default(devTools.REACT_DEVELOPER_TOOLS);

    this.props.notificationManager.addSuccess('🌈 Reload your window to start using the React dev tools!');
  }

  componentWillUnmount() {
    this.subscription.dispose();
  }

  componentDidUpdate() {
    this.subscription.dispose();
    this.subscription = new _eventKit.CompositeDisposable(this.props.repository.onMergeError(() => this.gitTabTracker.ensureVisible()));
  }

  onConsumeStatusBar(statusBar) {
    if (statusBar.disableGitInfoTile) {
      statusBar.disableGitInfoTile();
    }
  }

  clearGithubToken() {
    return this.props.loginModel.removeToken('https://api.github.com');
  }

  initializeRepo(initDialogPath) {
    if (this.state.initDialogActive) {
      return null;
    }

    if (!initDialogPath) {
      initDialogPath = this.props.repository.getWorkingDirectoryPath();
    }

    return new Promise(resolve => {
      this.setState({ initDialogActive: true, initDialogPath, initDialogResolve: resolve });
    });
  }

  showOpenIssueishDialog() {
    this.setState({ openIssueishDialogActive: true });
  }

  showWaterfallDiagnostics() {
    this.props.workspace.open(_gitTimingsView2.default.buildURI());
  }

  showCacheDiagnostics() {
    this.props.workspace.open(_gitCacheView2.default.buildURI());
  }

  async acceptClone(remoteUrl, projectPath) {
    this.setState({ cloneDialogInProgress: true });
    try {
      await this.props.cloneRepositoryForProjectPath(remoteUrl, projectPath);
      (0, _reporterProxy.addEvent)('clone-repo', { package: 'github' });
    } catch (e) {
      this.props.notificationManager.addError(`Unable to clone ${remoteUrl}`, { detail: e.stdErr, dismissable: true });
    } finally {
      this.setState({ cloneDialogInProgress: false, cloneDialogActive: false });
    }
  }

  cancelClone() {
    this.setState({ cloneDialogActive: false });
  }

  async acceptInit(projectPath) {
    try {
      await this.props.createRepositoryForProjectPath(projectPath);
      if (this.state.initDialogResolve) {
        this.state.initDialogResolve(projectPath);
      }
    } catch (e) {
      this.props.notificationManager.addError(`Unable to initialize git repository in ${projectPath}`, { detail: e.stdErr, dismissable: true });
    } finally {
      this.setState({ initDialogActive: false, initDialogPath: null, initDialogResolve: null });
    }
  }

  cancelInit() {
    if (this.state.initDialogResolve) {
      this.state.initDialogResolve(false);
    }
    this.setState({ initDialogActive: false, initDialogPath: null, initDialogResolve: null });
  }

  acceptOpenIssueish({ repoOwner, repoName, issueishNumber }) {
    const uri = _issueishDetailItem2.default.buildURI('https://api.github.com', repoOwner, repoName, issueishNumber);
    this.setState({ openIssueishDialogActive: false });
    this.props.workspace.open(uri).then(() => {
      (0, _reporterProxy.addEvent)('open-issueish-in-pane', { package: 'github', from: 'dialog' });
    });
  }

  cancelOpenIssueish() {
    this.setState({ openIssueishDialogActive: false });
  }

  destroyFilePatchPaneItems() {
    (0, _helpers.destroyFilePatchPaneItems)({ onlyStaged: false }, this.props.workspace);
  }

  destroyEmptyFilePatchPaneItems() {
    (0, _helpers.destroyEmptyFilePatchPaneItems)(this.props.workspace);
  }

  openCloneDialog() {
    this.setState({ cloneDialogActive: true });
  }

  quietlySelectItem(filePath, stagingStatus) {
    const gitTab = this.gitTabTracker.getComponent();
    return gitTab && gitTab.quietlySelectItem(filePath, stagingStatus);
  }

  async viewChangesForCurrentFile(stagingStatus) {
    const editor = this.props.workspace.getActiveTextEditor();
    if (!editor.getPath()) {
      return;
    }

    const absFilePath = await _fsExtra2.default.realpath(editor.getPath());
    const repoPath = this.props.repository.getWorkingDirectoryPath();
    if (repoPath === null) {
      const [projectPath] = this.props.project.relativizePath(editor.getPath());
      const notification = this.props.notificationManager.addInfo("Hmm, there's nothing to compare this file to", {
        description: 'You can create a Git repository to track changes to the files in your project.',
        dismissable: true,
        buttons: [{
          className: 'btn btn-primary',
          text: 'Create a repository now',
          onDidClick: async () => {
            notification.dismiss();
            const createdPath = await this.initializeRepo(projectPath);
            // If the user confirmed repository creation for this project path,
            // retry the operation that got them here in the first place
            if (createdPath === projectPath) {
              this.viewChangesForCurrentFile(stagingStatus);
            }
          }
        }]
      });
      return;
    }
    if (absFilePath.startsWith(repoPath)) {
      const filePath = absFilePath.slice(repoPath.length + 1);
      this.quietlySelectItem(filePath, stagingStatus);
      const splitDirection = this.props.config.get('github.viewChangesForCurrentFileDiffPaneSplitDirection');
      const pane = this.props.workspace.getActivePane();
      if (splitDirection === 'right') {
        pane.splitRight();
      } else if (splitDirection === 'down') {
        pane.splitDown();
      }
      const lineNum = editor.getCursorBufferPosition().row + 1;
      const item = await this.props.workspace.open(_changedFileItem2.default.buildURI(filePath, repoPath, stagingStatus), { pending: true, activatePane: true, activateItem: true });
      await item.getRealItemPromise();
      await item.getFilePatchLoadedPromise();
      item.goToDiffLine(lineNum);
      item.focus();
    } else {
      throw new Error(`${absFilePath} does not belong to repo ${repoPath}`);
    }
  }

  viewUnstagedChangesForCurrentFile() {
    return this.viewChangesForCurrentFile('unstaged');
  }

  viewStagedChangesForCurrentFile() {
    return this.viewChangesForCurrentFile('staged');
  }

  openFiles(filePaths, repository = this.props.repository) {
    return Promise.all(filePaths.map(filePath => {
      const absolutePath = _path2.default.join(repository.getWorkingDirectoryPath(), filePath);
      return this.props.workspace.open(absolutePath, { pending: filePaths.length === 1 });
    }));
  }

  getUnsavedFiles(filePaths, workdirPath) {
    const isModifiedByPath = new Map();
    this.props.workspace.getTextEditors().forEach(editor => {
      isModifiedByPath.set(editor.getPath(), editor.isModified());
    });
    return filePaths.filter(filePath => {
      const absFilePath = _path2.default.join(workdirPath, filePath);
      return isModifiedByPath.get(absFilePath);
    });
  }

  ensureNoUnsavedFiles(filePaths, message, workdirPath = this.props.repository.getWorkingDirectoryPath()) {
    const unsavedFiles = this.getUnsavedFiles(filePaths, workdirPath).map(filePath => `\`${filePath}\``).join('<br>');
    if (unsavedFiles.length) {
      this.props.notificationManager.addError(message, {
        description: `You have unsaved changes in:<br>${unsavedFiles}.`,
        dismissable: true
      });
      return false;
    } else {
      return true;
    }
  }

  async discardWorkDirChangesForPaths(filePaths) {
    const destructiveAction = () => {
      return this.props.repository.discardWorkDirChangesForPaths(filePaths);
    };
    return await this.props.repository.storeBeforeAndAfterBlobs(filePaths, () => this.ensureNoUnsavedFiles(filePaths, 'Cannot discard changes in selected files.'), destructiveAction);
  }

  async discardLines(multiFilePatch, lines, repository = this.props.repository) {
    // (kuychaco) For now we only support discarding rows for MultiFilePatches that contain a single file patch
    // The only way to access this method from the UI is to be in a ChangedFileItem, which only has a single file patch
    if (multiFilePatch.getFilePatches().length !== 1) {
      return Promise.resolve(null);
    }

    const filePath = multiFilePatch.getFilePatches()[0].getPath();
    const destructiveAction = async () => {
      const discardFilePatch = multiFilePatch.getUnstagePatchForLines(lines);
      await repository.applyPatchToWorkdir(discardFilePatch);
    };
    return await repository.storeBeforeAndAfterBlobs([filePath], () => this.ensureNoUnsavedFiles([filePath], 'Cannot discard lines.', repository.getWorkingDirectoryPath()), destructiveAction, filePath);
  }

  getFilePathsForLastDiscard(partialDiscardFilePath = null) {
    let lastSnapshots = this.props.repository.getLastHistorySnapshots(partialDiscardFilePath);
    if (partialDiscardFilePath) {
      lastSnapshots = lastSnapshots ? [lastSnapshots] : [];
    }
    return lastSnapshots.map(snapshot => snapshot.filePath);
  }

  async undoLastDiscard(partialDiscardFilePath = null, repository = this.props.repository) {
    const filePaths = this.getFilePathsForLastDiscard(partialDiscardFilePath);
    try {
      const results = await repository.restoreLastDiscardInTempFiles(() => this.ensureNoUnsavedFiles(filePaths, 'Cannot undo last discard.'), partialDiscardFilePath);
      if (results.length === 0) {
        return;
      }
      await this.proceedOrPromptBasedOnResults(results, partialDiscardFilePath);
    } catch (e) {
      if (e instanceof _gitShellOutStrategy.GitError && e.stdErr.match(/fatal: Not a valid object name/)) {
        this.cleanUpHistoryForFilePaths(filePaths, partialDiscardFilePath);
      } else {
        // eslint-disable-next-line no-console
        console.error(e);
      }
    }
  }

  async proceedOrPromptBasedOnResults(results, partialDiscardFilePath = null) {
    const conflicts = results.filter(({ conflict }) => conflict);
    if (conflicts.length === 0) {
      await this.proceedWithLastDiscardUndo(results, partialDiscardFilePath);
    } else {
      await this.promptAboutConflicts(results, conflicts, partialDiscardFilePath);
    }
  }

  async promptAboutConflicts(results, conflicts, partialDiscardFilePath = null) {
    const conflictedFiles = conflicts.map(({ filePath }) => `\t${filePath}`).join('\n');
    const choice = this.props.confirm({
      message: 'Undoing will result in conflicts...',
      detailedMessage: `for the following files:\n${conflictedFiles}\n` + 'Would you like to apply the changes with merge conflict markers, ' + 'or open the text with merge conflict markers in a new file?',
      buttons: ['Merge with conflict markers', 'Open in new file', 'Cancel']
    });
    if (choice === 0) {
      await this.proceedWithLastDiscardUndo(results, partialDiscardFilePath);
    } else if (choice === 1) {
      await this.openConflictsInNewEditors(conflicts.map(({ resultPath }) => resultPath));
    }
  }

  cleanUpHistoryForFilePaths(filePaths, partialDiscardFilePath = null) {
    this.props.repository.clearDiscardHistory(partialDiscardFilePath);
    const filePathsStr = filePaths.map(filePath => `\`${filePath}\``).join('<br>');
    this.props.notificationManager.addError('Discard history has expired.', {
      description: `Cannot undo discard for<br>${filePathsStr}<br>Stale discard history has been deleted.`,
      dismissable: true
    });
  }

  async proceedWithLastDiscardUndo(results, partialDiscardFilePath = null) {
    const promises = results.map(async result => {
      const { filePath, resultPath, deleted, conflict, theirsSha, commonBaseSha, currentSha } = result;
      const absFilePath = _path2.default.join(this.props.repository.getWorkingDirectoryPath(), filePath);
      if (deleted && resultPath === null) {
        await _fsExtra2.default.remove(absFilePath);
      } else {
        await _fsExtra2.default.copy(resultPath, absFilePath);
      }
      if (conflict) {
        await this.props.repository.writeMergeConflictToIndex(filePath, commonBaseSha, currentSha, theirsSha);
      }
    });
    await Promise.all(promises);
    await this.props.repository.popDiscardHistory(partialDiscardFilePath);
  }

  async openConflictsInNewEditors(resultPaths) {
    const editorPromises = resultPaths.map(resultPath => {
      return this.props.workspace.open(resultPath);
    });
    return await Promise.all(editorPromises);
  }

  /*
   * Asynchronously count the conflict markers present in a file specified by full path.
   */
  refreshResolutionProgress(fullPath) {
    const readStream = _fsExtra2.default.createReadStream(fullPath, { encoding: 'utf8' });
    return new Promise(resolve => {
      _conflict2.default.countFromStream(readStream).then(count => {
        this.props.resolutionProgress.reportMarkerCount(fullPath, count);
      });
    });
  }

  /*
   * Display the credential entry dialog. Return a Promise that will resolve with the provided credentials on accept
   * or reject on cancel.
   */
  promptForCredentials(query) {
    return new Promise((resolve, reject) => {
      this.setState({
        credentialDialogQuery: _extends({}, query, {
          onSubmit: response => this.setState({ credentialDialogQuery: null }, () => resolve(response)),
          onCancel: () => this.setState({ credentialDialogQuery: null }, reject)
        })
      });
    });
  }
}

exports.default = RootController;
RootController.propTypes = {
  workspace: _propTypes2.default.object.isRequired,
  commandRegistry: _propTypes2.default.object.isRequired,
  deserializers: _propTypes2.default.object.isRequired,
  notificationManager: _propTypes2.default.object.isRequired,
  tooltips: _propTypes2.default.object.isRequired,
  keymaps: _propTypes2.default.object.isRequired,
  grammars: _propTypes2.default.object.isRequired,
  config: _propTypes2.default.object.isRequired,
  project: _propTypes2.default.object.isRequired,
  loginModel: _propTypes2.default.object.isRequired,
  confirm: _propTypes2.default.func.isRequired,
  createRepositoryForProjectPath: _propTypes2.default.func,
  cloneRepositoryForProjectPath: _propTypes2.default.func,
  repository: _propTypes2.default.object.isRequired,
  resolutionProgress: _propTypes2.default.object.isRequired,
  statusBar: _propTypes2.default.object,
  switchboard: _propTypes2.default.instanceOf(_switchboard2.default),
  startOpen: _propTypes2.default.bool,
  startRevealed: _propTypes2.default.bool,
  pipelineManager: _propTypes2.default.object,
  workdirContextPool: _propTypes3.WorkdirContextPoolPropType.isRequired
};
RootController.defaultProps = {
  switchboard: new _switchboard2.default(),
  startOpen: false,
  startRevealed: false
};
class TabTracker {
  constructor(name, { getWorkspace, uri }) {
    (0, _helpers.autobind)(this, 'toggle', 'toggleFocus', 'ensureVisible');
    this.name = name;

    this.getWorkspace = getWorkspace;
    this.uri = uri;
  }

  async toggle() {
    const focusToRestore = document.activeElement;
    let shouldRestoreFocus = false;

    // Rendered => the dock item is being rendered, whether or not the dock is visible or the item
    //   is visible within its dock.
    // Visible => the item is active and the dock item is active within its dock.
    const wasRendered = this.isRendered();
    const wasVisible = this.isVisible();

    if (!wasRendered || !wasVisible) {
      // Not rendered, or rendered but not an active item in a visible dock.
      await this.reveal();
      shouldRestoreFocus = true;
    } else {
      // Rendered and an active item within a visible dock.
      await this.hide();
      shouldRestoreFocus = false;
    }

    if (shouldRestoreFocus) {
      process.nextTick(() => focusToRestore.focus());
    }
  }

  async toggleFocus() {
    const hadFocus = this.hasFocus();
    await this.ensureVisible();

    if (hadFocus) {
      let workspace = this.getWorkspace();
      if (workspace.getCenter) {
        workspace = workspace.getCenter();
      }
      workspace.getActivePane().activate();
    } else {
      this.focus();
    }
  }

  async ensureVisible() {
    if (!this.isVisible()) {
      await this.reveal();
      return true;
    }
    return false;
  }

  ensureRendered() {
    return this.getWorkspace().open(this.uri, { searchAllPanes: true, activateItem: false, activatePane: false });
  }

  reveal() {
    (0, _reporterProxy.incrementCounter)(`${this.name}-tab-open`);
    return this.getWorkspace().open(this.uri, { searchAllPanes: true, activateItem: true, activatePane: true });
  }

  hide() {
    (0, _reporterProxy.incrementCounter)(`${this.name}-tab-close`);
    return this.getWorkspace().hide(this.uri);
  }

  focus() {
    this.getComponent().restoreFocus();
  }

  getItem() {
    const pane = this.getWorkspace().paneForURI(this.uri);
    if (!pane) {
      return null;
    }

    const paneItem = pane.itemForURI(this.uri);
    if (!paneItem) {
      return null;
    }

    return paneItem;
  }

  getComponent() {
    const paneItem = this.getItem();
    if (!paneItem) {
      return null;
    }
    if (typeof paneItem.getRealItem !== 'function') {
      return null;
    }

    return paneItem.getRealItem();
  }

  getDOMElement() {
    const paneItem = this.getItem();
    if (!paneItem) {
      return null;
    }
    if (typeof paneItem.getElement !== 'function') {
      return null;
    }

    return paneItem.getElement();
  }

  isRendered() {
    return !!this.getWorkspace().paneForURI(this.uri);
  }

  isVisible() {
    const workspace = this.getWorkspace();
    return workspace.getPaneContainers().filter(container => container === workspace.getCenter() || container.isVisible()).some(container => container.getPanes().some(pane => {
      const item = pane.getActiveItem();
      return item && item.getURI && item.getURI() === this.uri;
    }));
  }

  hasFocus() {
    const root = this.getDOMElement();
    return root && root.contains(document.activeElement);
  }
}
//# sourceMappingURL=data:application/json;charset=utf-8;base64,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