'use strict';

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.BareIssueishDetailController = undefined;

var _react = require('react');

var _react2 = _interopRequireDefault(_react);

var _reactRelay = require('react-relay');

var _propTypes = require('prop-types');

var _propTypes2 = _interopRequireDefault(_propTypes);

var _propTypes3 = require('../prop-types');

var _gitShellOutStrategy = require('../git-shell-out-strategy');

var _enableableOperation = require('../models/enableable-operation');

var _enableableOperation2 = _interopRequireDefault(_enableableOperation);

var _issueishDetailView = require('../views/issueish-detail-view');

var _issueishDetailView2 = _interopRequireDefault(_issueishDetailView);

var _reporterProxy = require('../reporter-proxy');

function _interopRequireDefault(obj) { return obj && obj.__esModule ? obj : { default: obj }; }

class BareIssueishDetailController extends _react2.default.Component {

  constructor(props) {
    super(props);

    this.state = {
      checkoutInProgress: false
    };

    this.checkoutOp = new _enableableOperation2.default(() => this.checkout().catch(e => {
      if (!(e instanceof _gitShellOutStrategy.GitError)) {
        throw e;
      }
    }));
    this.checkoutOp.toggleState(this, 'checkoutInProgress');
  }

  componentDidMount() {
    this.updateTitle();
  }

  componentDidUpdate() {
    this.updateTitle();
  }

  updateTitle() {
    const { repository } = this.props;
    if (repository && repository.issueish) {
      const issueish = repository.issueish;
      const prefix = issueish.__typename === 'Issue' ? 'Issue:' : 'PR:';

      const title = `${prefix} ${repository.owner.login}/${repository.name}#${issueish.number} — ${issueish.title}`;
      this.props.onTitleChange(title);
    }
  }

  render() {
    const { repository } = this.props;
    if (!repository || !repository.issueish) {
      return _react2.default.createElement(
        'div',
        null,
        'Issue/PR #',
        this.props.issueishNumber,
        ' not found'
      ); // TODO: no PRs
    }

    this.checkoutOp = this.nextCheckoutOp();

    return _react2.default.createElement(_issueishDetailView2.default, {
      repository: repository,
      issueish: repository.issueish,
      checkoutOp: this.checkoutOp,
      switchToIssueish: this.props.switchToIssueish
    });
  }

  nextCheckoutOp() {
    const { repository } = this.props;
    const { issueish } = repository;

    if (issueish.__typename !== 'PullRequest') {
      return this.checkoutOp.disable(_issueishDetailView.checkoutStates.HIDDEN, 'Cannot check out an issue');
    }

    if (this.props.isAbsent) {
      return this.checkoutOp.disable(_issueishDetailView.checkoutStates.HIDDEN, 'No repository found');
    }

    if (this.props.isLoading) {
      return this.checkoutOp.disable(_issueishDetailView.checkoutStates.DISABLED, 'Loading');
    }

    if (!this.props.isPresent) {
      return this.checkoutOp.disable(_issueishDetailView.checkoutStates.DISABLED, 'No repository found');
    }

    if (this.props.isMerging) {
      return this.checkoutOp.disable(_issueishDetailView.checkoutStates.DISABLED, 'Merge in progress');
    }

    if (this.props.isRebasing) {
      return this.checkoutOp.disable(_issueishDetailView.checkoutStates.DISABLED, 'Rebase in progress');
    }

    if (this.state.checkoutInProgress) {
      return this.checkoutOp.disable(_issueishDetailView.checkoutStates.DISABLED, 'Checking out...');
    }

    // determine if pullRequest.headRepository is null
    // this can happen if a repository has been deleted.
    if (!issueish.headRepository) {
      return this.checkoutOp.disable(_issueishDetailView.checkoutStates.DISABLED, 'Pull request head repository does not exist');
    }

    // Determine if we already have this PR checked out.

    const headPush = this.props.branches.getHeadBranch().getPush();
    const headRemote = this.props.remotes.withName(headPush.getRemoteName());

    // (detect checkout from pull/### refspec)
    const fromPullRefspec = headRemote.getOwner() === repository.owner.login && headRemote.getRepo() === repository.name && headPush.getShortRemoteRef() === `pull/${issueish.number}/head`;

    // (detect checkout from head repository)
    const fromHeadRepo = headRemote.getOwner() === issueish.headRepository.owner.login && headRemote.getRepo() === issueish.headRepository.name && headPush.getShortRemoteRef() === issueish.headRefName;

    if (fromPullRefspec || fromHeadRepo) {
      return this.checkoutOp.disable(_issueishDetailView.checkoutStates.CURRENT, 'Current');
    }

    return this.checkoutOp.enable();
  }

  async checkout() {
    const { repository } = this.props;
    const { issueish } = repository;
    const { headRepository } = issueish;

    const fullHeadRef = `refs/heads/${issueish.headRefName}`;

    let sourceRemoteName, localRefName;

    // Discover or create a remote pointing to the repo containing the pull request's head ref.
    // If the local repository already has the head repository specified as a remote, that remote will be used, so
    // that any related configuration is picked up for the fetch. Otherwise, the head repository fetch URL is used
    // directly.
    const headRemotes = this.props.remotes.matchingGitHubRepository(headRepository.owner.login, headRepository.name);
    if (headRemotes.length > 0) {
      sourceRemoteName = headRemotes[0].getName();
    } else {
      const url = {
        https: headRepository.url + '.git',
        ssh: headRepository.sshUrl
      }[this.props.remotes.mostUsedProtocol(['https', 'ssh'])];

      // This will throw if a remote with this name already exists (and points somewhere else, or we would have found
      // it above). ¯\_(ツ)_/¯
      const remote = await this.props.addRemote(headRepository.owner.login, url);
      sourceRemoteName = remote.getName();
    }

    // Identify an existing local ref that already corresponds to the pull request, if one exists. Otherwise, generate
    // a new local ref name.
    const pullTargets = this.props.branches.getPullTargets(sourceRemoteName, fullHeadRef);
    if (pullTargets.length > 0) {
      localRefName = pullTargets[0].getName();

      // Check out the existing local ref.
      await this.props.checkout(localRefName);
      try {
        await this.props.pull(fullHeadRef, { remoteName: sourceRemoteName, ffOnly: true });
      } finally {
        (0, _reporterProxy.incrementCounter)('checkout-pr');
      }

      return;
    }

    await this.props.fetch(fullHeadRef, { remoteName: sourceRemoteName });

    // Check out the local ref and set it up to track the head ref.
    await this.props.checkout(`pr-${issueish.number}/${headRepository.owner.login}/${issueish.headRefName}`, {
      createNew: true,
      track: true,
      startPoint: `refs/remotes/${sourceRemoteName}/${issueish.headRefName}`
    });

    (0, _reporterProxy.incrementCounter)('checkout-pr');
  }
}

exports.BareIssueishDetailController = BareIssueishDetailController;
BareIssueishDetailController.propTypes = {
  repository: _propTypes2.default.shape({
    name: _propTypes2.default.string.isRequired,
    owner: _propTypes2.default.shape({
      login: _propTypes2.default.string.isRequired
    }).isRequired,
    issueish: _propTypes2.default.any // FIXME from IssueishPaneItemContainer.propTypes
  }),
  issueishNumber: _propTypes2.default.number.isRequired,

  branches: _propTypes3.BranchSetPropType.isRequired,
  remotes: _propTypes3.RemoteSetPropType.isRequired,
  isMerging: _propTypes2.default.bool.isRequired,
  isRebasing: _propTypes2.default.bool.isRequired,
  isAbsent: _propTypes2.default.bool.isRequired,
  isLoading: _propTypes2.default.bool.isRequired,
  isPresent: _propTypes2.default.bool.isRequired,

  fetch: _propTypes2.default.func.isRequired,
  checkout: _propTypes2.default.func.isRequired,
  pull: _propTypes2.default.func.isRequired,
  addRemote: _propTypes2.default.func.isRequired,
  onTitleChange: _propTypes2.default.func.isRequired,
  switchToIssueish: _propTypes2.default.func.isRequired
};
exports.default = (0, _reactRelay.createFragmentContainer)(BareIssueishDetailController, {
  repository: function () {
    const node = require('./__generated__/issueishDetailController_repository.graphql');

    if (node.hash && node.hash !== '85f9f003c5256db379ff5b0bdf7794a4') {
      console.error('The definition of \'issueishDetailController_repository\' appears to have changed. Run `relay-compiler` to update the generated files to receive the expected data.');
    }

    return require('./__generated__/issueishDetailController_repository.graphql');
  }
});
//# sourceMappingURL=data:application/json;charset=utf-8;base64,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