'use strict';

Object.defineProperty(exports, "__esModule", {
  value: true
});

var _eventKit = require('event-kit');

var _react = require('react');

var _react2 = _interopRequireDefault(_react);

var _propTypes = require('prop-types');

var _propTypes2 = _interopRequireDefault(_propTypes);

var _compareSets = require('compare-sets');

var _compareSets2 = _interopRequireDefault(_compareSets);

var _commands = require('../atom/commands');

var _commands2 = _interopRequireDefault(_commands);

var _conflict = require('../models/conflicts/conflict');

var _conflict2 = _interopRequireDefault(_conflict);

var _conflictController = require('./conflict-controller');

var _conflictController2 = _interopRequireDefault(_conflictController);

var _source = require('../models/conflicts/source');

var _helpers = require('../helpers');

function _interopRequireDefault(obj) { return obj && obj.__esModule ? obj : { default: obj }; }

/**
 * Render a `ConflictController` for each conflict marker within an open TextEditor.
 */
class EditorConflictController extends _react2.default.Component {

  constructor(props, context) {
    super(props, context);
    (0, _helpers.autobind)(this, 'resolveAsCurrent', 'revertConflictModifications', 'dismissCurrent');

    // this.layer = props.editor.addMarkerLayer({
    //   maintainHistory: true,
    //   persistent: false,
    // });

    this.layer = props.editor.getDefaultMarkerLayer();

    this.state = {
      conflicts: new Set(_conflict2.default.allFromEditor(props.editor, this.layer, props.isRebase))
    };

    this.subscriptions = new _eventKit.CompositeDisposable();

    this.updateMarkerCount();
  }

  componentDidMount() {
    const buffer = this.props.editor.getBuffer();

    this.subscriptions.add(this.props.editor.onDidStopChanging(() => this.forceUpdate()), this.props.editor.onDidDestroy(() => this.props.refreshResolutionProgress(this.props.editor.getPath())), buffer.onDidReload(() => this.reparseConflicts()));
  }

  render() {
    this.updateMarkerCount();

    return _react2.default.createElement(
      'div',
      null,
      this.state.conflicts.size > 0 && _react2.default.createElement(
        _commands2.default,
        { registry: this.props.commandRegistry, target: 'atom-text-editor' },
        _react2.default.createElement(_commands.Command, { command: 'github:resolve-as-ours', callback: this.getResolverUsing([_source.OURS]) }),
        _react2.default.createElement(_commands.Command, { command: 'github:resolve-as-theirs', callback: this.getResolverUsing([_source.THEIRS]) }),
        _react2.default.createElement(_commands.Command, { command: 'github:resolve-as-base', callback: this.getResolverUsing([_source.BASE]) }),
        _react2.default.createElement(_commands.Command, { command: 'github:resolve-as-ours-then-theirs', callback: this.getResolverUsing([_source.OURS, _source.THEIRS]) }),
        _react2.default.createElement(_commands.Command, { command: 'github:resolve-as-theirs-then-ours', callback: this.getResolverUsing([_source.THEIRS, _source.OURS]) }),
        _react2.default.createElement(_commands.Command, { command: 'github:resolve-as-current', callback: this.resolveAsCurrent }),
        _react2.default.createElement(_commands.Command, { command: 'github:revert-conflict-modifications', callback: this.revertConflictModifications }),
        _react2.default.createElement(_commands.Command, { command: 'github:dismiss-conflict', callback: this.dismissCurrent })
      ),
      Array.from(this.state.conflicts, c => _react2.default.createElement(_conflictController2.default, {
        key: c.getKey(),
        editor: this.props.editor,
        conflict: c,
        resolveAsSequence: sources => this.resolveAsSequence(c, sources),
        dismiss: () => this.dismissConflicts([c])
      }))
    );
  }

  componentWillUnmount() {
    // this.layer.destroy();
    this.subscriptions.dispose();
  }

  /*
   * Return an Array containing `Conflict` objects whose marked regions include any cursor position in the current
   * `TextEditor` and the `Sides` that contain a cursor within each.
   *
   * This method is written to have linear complexity with respect to the number of cursors and the number of
   * conflicts, to gracefully handle files with large numbers of both.
   */
  getCurrentConflicts() {
    const cursorPositions = this.props.editor.getCursorBufferPositions();
    cursorPositions.sort((a, b) => a.compare(b));
    const cursorIterator = cursorPositions[Symbol.iterator]();

    const conflictIterator = this.state.conflicts.keys();

    let currentCursor = cursorIterator.next();
    let currentConflict = conflictIterator.next();
    const activeConflicts = [];

    while (!currentCursor.done && !currentConflict.done) {
      // Advance currentCursor to the first cursor beyond the earliest conflict.
      const earliestConflictPosition = currentConflict.value.getRange().start;
      while (!currentCursor.done && currentCursor.value.isLessThan(earliestConflictPosition)) {
        currentCursor = cursorIterator.next();
      }

      // Advance currentConflict until the first conflict that begins at a position after the current cursor.
      // Compare each to the current cursor, and add it to activeConflicts if it contains it.
      while (!currentConflict.done && !currentCursor.done && currentConflict.value.getRange().start.isLessThan(currentCursor.value)) {
        if (currentConflict.value.includesPoint(currentCursor.value)) {
          // Hit; determine which sides of this conflict contain cursors.
          const conflict = currentConflict.value;
          const endPosition = conflict.getRange().end;
          const sides = new Set();
          while (!currentCursor.done && currentCursor.value.isLessThan(endPosition)) {
            const side = conflict.getSideContaining(currentCursor.value);
            if (side) {
              sides.add(side);
            }
            currentCursor = cursorIterator.next();
          }

          activeConflicts.push({ conflict, sides });
        }

        currentConflict = conflictIterator.next();
      }
    }

    return activeConflicts;
  }

  getResolverUsing(sequence) {
    return () => {
      this.getCurrentConflicts().forEach(match => this.resolveAsSequence(match.conflict, sequence));
    };
  }

  resolveAsCurrent() {
    this.getCurrentConflicts().forEach(match => {
      if (match.sides.size === 1) {
        const side = match.sides.keys().next().value;
        this.resolveAs(match.conflict, side.getSource());
      }
    });
  }

  revertConflictModifications() {
    this.getCurrentConflicts().forEach(match => {
      match.sides.forEach(side => {
        side.isModified() && side.revert();
        side.isBannerModified() && side.revertBanner();
      });
    });
  }

  dismissCurrent() {
    this.dismissConflicts(this.getCurrentConflicts().map(match => match.conflict));
  }

  dismissConflicts(conflicts) {
    this.setState((prevState, props) => {
      const { added } = (0, _compareSets2.default)(new Set(conflicts), prevState.conflicts);
      return { conflicts: added };
    });
  }

  resolveAsSequence(conflict, sources) {
    const [firstSide, ...restOfSides] = sources.map(source => conflict.getSide(source)).filter(side => side);

    const textToAppend = restOfSides.map(side => side.getText()).join('');

    this.props.editor.transact(() => {
      // Append text from all but the first Side to the first Side. Adjust the following DisplayMarker so that only that
      // Side's marker includes the appended text, not the next one.
      const appendedRange = firstSide.appendText(textToAppend);
      const nextMarker = conflict.markerAfter(firstSide.getPosition());
      if (nextMarker) {
        nextMarker.setTailBufferPosition(appendedRange.end);
      }

      this.innerResolveAs(conflict, sources[0]);
    });
  }

  resolveAs(conflict, source) {
    this.props.editor.transact(() => {
      this.innerResolveAs(conflict, source);
    });
  }

  innerResolveAs(conflict, source) {
    conflict.resolveAs(source);

    const chosenSide = conflict.getChosenSide();
    if (!chosenSide.isBannerModified()) {
      chosenSide.deleteBanner();
    }

    const separator = conflict.getSeparator();
    if (!separator.isModified()) {
      separator.delete();
    }

    conflict.getUnchosenSides().forEach(side => {
      side.deleteBanner();
      side.delete();
    });

    this.updateMarkerCount();
  }

  reparseConflicts() {
    const newConflicts = new Set(_conflict2.default.allFromEditor(this.props.editor, this.layer, this.props.isRebase));
    this.setState({ conflicts: newConflicts });
  }

  updateMarkerCount() {
    this.props.resolutionProgress.reportMarkerCount(this.props.editor.getPath(), Array.from(this.state.conflicts, c => !c.isResolved()).filter(b => b).length);
  }
}
exports.default = EditorConflictController;
EditorConflictController.propTypes = {
  editor: _propTypes2.default.object.isRequired,
  commandRegistry: _propTypes2.default.object.isRequired,
  resolutionProgress: _propTypes2.default.object.isRequired,
  isRebase: _propTypes2.default.bool.isRequired,
  refreshResolutionProgress: _propTypes2.default.func
};
EditorConflictController.defaultProps = {
  refreshResolutionProgress: () => {}
};
//# sourceMappingURL=data:application/json;charset=utf-8;base64,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