'use strict';

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.COMMIT_GRAMMAR_SCOPE = undefined;

var _extends = Object.assign || function (target) { for (var i = 1; i < arguments.length; i++) { var source = arguments[i]; for (var key in source) { if (Object.prototype.hasOwnProperty.call(source, key)) { target[key] = source[key]; } } } return target; };

var _path = require('path');

var _path2 = _interopRequireDefault(_path);

var _atom = require('atom');

var _react = require('react');

var _react2 = _interopRequireDefault(_react);

var _propTypes = require('prop-types');

var _propTypes2 = _interopRequireDefault(_propTypes);

var _eventKit = require('event-kit');

var _fsExtra = require('fs-extra');

var _fsExtra2 = _interopRequireDefault(_fsExtra);

var _commitView = require('../views/commit-view');

var _commitView2 = _interopRequireDefault(_commitView);

var _refHolder = require('../models/ref-holder');

var _refHolder2 = _interopRequireDefault(_refHolder);

var _commitPreviewItem = require('../items/commit-preview-item');

var _commitPreviewItem2 = _interopRequireDefault(_commitPreviewItem);

var _propTypes3 = require('../prop-types');

var _watchWorkspaceItem = require('../watch-workspace-item');

var _helpers = require('../helpers');

var _reporterProxy = require('../reporter-proxy');

function _interopRequireDefault(obj) { return obj && obj.__esModule ? obj : { default: obj }; }

const COMMIT_GRAMMAR_SCOPE = exports.COMMIT_GRAMMAR_SCOPE = 'text.git-commit';

class CommitController extends _react2.default.Component {

  constructor(props, context) {
    super(props, context);

    this.activateCommitPreview = () => {
      const uri = _commitPreviewItem2.default.buildURI(this.props.repository.getWorkingDirectoryPath());
      return this.props.workspace.open(uri, { searchAllPanes: true, pending: true, activate: true });
    };

    (0, _helpers.autobind)(this, 'commit', 'handleMessageChange', 'toggleExpandedCommitMessageEditor', 'grammarAdded', 'toggleCommitPreview');

    this.subscriptions = new _eventKit.CompositeDisposable();
    this.refCommitView = new _refHolder2.default();

    this.commitMessageBuffer = new _atom.TextBuffer({ text: this.props.repository.getCommitMessage() });
    this.subscriptions.add(this.commitMessageBuffer.onDidChange(this.handleMessageChange));

    this.previewWatcher = (0, _watchWorkspaceItem.watchWorkspaceItem)(this.props.workspace, _commitPreviewItem2.default.buildURI(this.props.repository.getWorkingDirectoryPath()), this, 'commitPreviewActive');
    this.subscriptions.add(this.previewWatcher);
  }

  componentDidMount() {
    this.subscriptions.add(this.props.workspace.onDidAddTextEditor(({ textEditor }) => {
      if (this.props.repository.isPresent() && textEditor.getPath() === this.getCommitMessagePath()) {
        const grammar = this.props.grammars.grammarForScopeName(COMMIT_GRAMMAR_SCOPE);
        if (grammar) {
          textEditor.setGrammar(grammar);
        }
      }
    }), this.props.workspace.onDidDestroyPaneItem(async ({ item }) => {
      if (this.props.repository.isPresent() && item.getPath && item.getPath() === this.getCommitMessagePath() && this.getCommitMessageEditors().length === 0) {
        // we closed the last editor pointing to the commit message file
        try {
          this.commitMessageBuffer.setText((await _fsExtra2.default.readFile(this.getCommitMessagePath(), { encoding: 'utf8' })));
        } catch (e) {
          if (e.code !== 'ENOENT') {
            throw e;
          }
        }
      }
    }));
  }

  render() {
    const operationStates = this.props.repository.getOperationStates();

    return _react2.default.createElement(_commitView2.default, {
      ref: this.refCommitView.setter,
      workspace: this.props.workspace,
      tooltips: this.props.tooltips,
      config: this.props.config,
      stagedChangesExist: this.props.stagedChangesExist,
      mergeConflictsExist: this.props.mergeConflictsExist,
      prepareToCommit: this.props.prepareToCommit,
      commit: this.commit,
      abortMerge: this.props.abortMerge,
      commandRegistry: this.props.commandRegistry,
      maximumCharacterLimit: 72,
      messageBuffer: this.commitMessageBuffer,
      isMerging: this.props.isMerging,
      isCommitting: operationStates.isCommitInProgress(),
      lastCommit: this.props.lastCommit,
      currentBranch: this.props.currentBranch,
      toggleExpandedCommitMessageEditor: this.toggleExpandedCommitMessageEditor,
      deactivateCommitBox: this.isCommitMessageEditorExpanded(),
      userStore: this.props.userStore,
      selectedCoAuthors: this.props.selectedCoAuthors,
      updateSelectedCoAuthors: this.props.updateSelectedCoAuthors,
      toggleCommitPreview: this.toggleCommitPreview,
      activateCommitPreview: this.activateCommitPreview,
      commitPreviewActive: this.state.commitPreviewActive
    });
  }

  componentDidUpdate(prevProps) {
    this.commitMessageBuffer.setTextViaDiff(this.getCommitMessage());

    if (prevProps.repository !== this.props.repository) {
      this.previewWatcher.setPattern(_commitPreviewItem2.default.buildURI(this.props.repository.getWorkingDirectoryPath()));
    }
  }

  componentWillUnmount() {
    this.subscriptions.dispose();
  }

  commit(message, coAuthors = [], amend = false) {
    let msg, verbatim;
    if (this.isCommitMessageEditorExpanded()) {
      msg = this.getCommitMessageEditors()[0].getText();
      verbatim = false;
    } else {
      const wrapMessage = this.props.config.get('github.automaticCommitMessageWrapping');
      msg = wrapMessage ? wrapCommitMessage(message) : message;
      verbatim = true;
    }

    return this.props.commit(msg.trim(), { amend, coAuthors, verbatim });
  }

  setCommitMessage(message, options) {
    if (!this.props.repository.isPresent()) {
      return;
    }
    this.props.repository.setCommitMessage(message, options);
  }

  getCommitMessage() {
    return this.props.repository.getCommitMessage();
  }

  getCommitMessagePath() {
    return _path2.default.join(this.props.repository.getGitDirectoryPath(), 'ATOM_COMMIT_EDITMSG');
  }

  handleMessageChange() {
    if (!this.props.repository.isPresent()) {
      return;
    }
    this.setCommitMessage(this.commitMessageBuffer.getText(), { suppressUpdate: true });
  }

  getCommitMessageEditors() {
    if (!this.props.repository.isPresent()) {
      return [];
    }
    return this.props.workspace.getTextEditors().filter(editor => editor.getPath() === this.getCommitMessagePath());
  }

  async toggleExpandedCommitMessageEditor(messageFromBox) {
    if (this.isCommitMessageEditorExpanded()) {
      if (this.commitMessageEditorIsInForeground()) {
        await this.closeAllOpenCommitMessageEditors();
        this.forceUpdate();
      } else {
        this.activateCommitMessageEditor();
      }
    } else {
      await this.openCommitMessageEditor(messageFromBox);
      this.forceUpdate();
    }
  }

  isCommitMessageEditorExpanded() {
    return this.getCommitMessageEditors().length > 0;
  }

  commitMessageEditorIsInForeground() {
    const commitMessageEditorsInForeground = this.props.workspace.getPanes().map(pane => pane.getActiveItem()).filter(item => item && item.getPath && item.getPath() === this.getCommitMessagePath());
    return commitMessageEditorsInForeground.length > 0;
  }

  activateCommitMessageEditor() {
    const panes = this.props.workspace.getPanes();
    let editor;
    const paneWithEditor = panes.find(pane => {
      editor = pane.getItems().find(item => item.getPath && item.getPath() === this.getCommitMessagePath());
      return !!editor;
    });
    paneWithEditor.activate();
    paneWithEditor.activateItem(editor);
  }

  closeAllOpenCommitMessageEditors() {
    return Promise.all(this.props.workspace.getPanes().map(pane => {
      return Promise.all(pane.getItems().map(async item => {
        if (item && item.getPath && item.getPath() === this.getCommitMessagePath()) {
          const destroyed = await pane.destroyItem(item);
          if (!destroyed) {
            pane.activateItem(item);
          }
        }
      }));
    }));
  }

  async openCommitMessageEditor(messageFromBox) {
    await _fsExtra2.default.writeFile(this.getCommitMessagePath(), messageFromBox, 'utf8');
    const commitEditor = await this.props.workspace.open(this.getCommitMessagePath());
    (0, _reporterProxy.addEvent)('open-commit-message-editor', { package: 'github' });

    const grammar = this.props.grammars.grammarForScopeName(COMMIT_GRAMMAR_SCOPE);
    if (grammar) {
      commitEditor.setGrammar(grammar);
    } else {
      this.grammarSubscription = this.props.grammars.onDidAddGrammar(this.grammarAdded);
      this.subscriptions.add(this.grammarSubscription);
    }
  }

  grammarAdded(grammar) {
    if (grammar.scopeName !== COMMIT_GRAMMAR_SCOPE) {
      return;
    }

    this.getCommitMessageEditors().forEach(editor => editor.setGrammar(grammar));
    this.grammarSubscription.dispose();
  }

  rememberFocus(event) {
    return this.refCommitView.map(view => view.rememberFocus(event)).getOr(null);
  }

  setFocus(focus) {
    return this.refCommitView.map(view => view.setFocus(focus)).getOr(false);
  }

  advanceFocus(...args) {
    return this.refCommitView.map(view => view.advanceFocus(...args)).getOr(false);
  }

  retreatFocus(...args) {
    return this.refCommitView.map(view => view.retreatFocus(...args)).getOr(false);
  }

  hasFocusAtBeginning() {
    return this.refCommitView.map(view => view.hasFocusAtBeginning()).getOr(false);
  }

  toggleCommitPreview() {
    (0, _reporterProxy.addEvent)('toggle-commit-preview', { package: 'github' });
    const uri = _commitPreviewItem2.default.buildURI(this.props.repository.getWorkingDirectoryPath());
    if (this.props.workspace.hide(uri)) {
      return Promise.resolve();
    } else {
      return this.props.workspace.open(uri, { searchAllPanes: true, pending: true });
    }
  }

}

exports.default = CommitController;
CommitController.focus = _extends({}, _commitView2.default.focus);
CommitController.propTypes = {
  workspace: _propTypes2.default.object.isRequired,
  grammars: _propTypes2.default.object.isRequired,
  commandRegistry: _propTypes2.default.object.isRequired,
  config: _propTypes2.default.object.isRequired,
  tooltips: _propTypes2.default.object.isRequired,

  repository: _propTypes2.default.object.isRequired,
  isMerging: _propTypes2.default.bool.isRequired,
  mergeConflictsExist: _propTypes2.default.bool.isRequired,
  stagedChangesExist: _propTypes2.default.bool.isRequired,
  lastCommit: _propTypes2.default.object.isRequired,
  currentBranch: _propTypes2.default.object.isRequired,
  userStore: _propTypes3.UserStorePropType.isRequired,
  selectedCoAuthors: _propTypes2.default.arrayOf(_propTypes3.AuthorPropType),
  updateSelectedCoAuthors: _propTypes2.default.func,
  prepareToCommit: _propTypes2.default.func.isRequired,
  commit: _propTypes2.default.func.isRequired,
  abortMerge: _propTypes2.default.func.isRequired
};
function wrapCommitMessage(message) {
  // hard wrap message (except for first line) at 72 characters
  let results = [];
  message.split('\n').forEach((line, index) => {
    if (line.length <= 72 || index === 0) {
      results.push(line);
    } else {
      const matches = line.match(/.{1,72}(\s|$)|\S+?(\s|$)/g).map(match => {
        return match.endsWith('\n') ? match.substr(0, match.length - 1) : match;
      });
      results = results.concat(matches);
    }
  });

  return results.join('\n');
}
//# sourceMappingURL=data:application/json;charset=utf-8;base64,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