'use strict';

Object.defineProperty(exports, "__esModule", {
  value: true
});

var _react = require('react');

var _react2 = _interopRequireDefault(_react);

var _reactDom = require('react-dom');

var _reactDom2 = _interopRequireDefault(_reactDom);

var _propTypes = require('prop-types');

var _propTypes2 = _interopRequireDefault(_propTypes);

var _eventKit = require('event-kit');

var _uriPattern = require('./uri-pattern');

var _uriPattern2 = _interopRequireDefault(_uriPattern);

var _refHolder = require('../models/ref-holder');

var _refHolder2 = _interopRequireDefault(_refHolder);

var _stubItem = require('../items/stub-item');

var _stubItem2 = _interopRequireDefault(_stubItem);

var _helpers = require('../helpers');

function _interopRequireDefault(obj) { return obj && obj.__esModule ? obj : { default: obj }; }

/**
 * PaneItem registers an opener with the current Atom workspace as long as this component is mounted. The opener will
 * trigger on URIs that match a specified pattern and render a subtree returned by a render prop.
 *
 * The render prop can receive three arguments:
 *
 * * itemHolder: A RefHolder. If used as the target for a ref, the referenced component will be used as the "item" of
 *   the pane item - its `getTitle()`, `getIconName()`, and other methods will be used by the pane.
 * * params: An object containing the named parameters captured by the URI pattern.
 * * uri: The exact, matched URI used to launch this item.
 *
 * render() {
 *   return (
 *     <PaneItem workspace={this.props.workspace} uriPattern='atom-github://host/{id}'>
 *       {({itemHolder, params}) => (
 *         <ItemComponent ref={itemHolder.setter} id={params.id} />
 *       )}
 *     </PaneItem>
 *   );
 * }
 */
class PaneItem extends _react2.default.Component {

  constructor(props) {
    super(props);
    (0, _helpers.autobind)(this, 'opener');

    const uriPattern = new _uriPattern2.default(this.props.uriPattern);
    const currentlyOpen = this.props.workspace.getPaneItems().reduce((arr, item) => {
      const element = item.getElement ? item.getElement() : null;
      const match = item.getURI ? uriPattern.matches(item.getURI()) : _uriPattern.nonURIMatch;
      const stub = item.setRealItem ? item : null;

      if (element && match.ok()) {
        const openItem = new OpenItem(match, element, stub);
        arr.push(openItem);
      }

      return arr;
    }, []);

    this.subs = new _eventKit.CompositeDisposable();
    this.state = { uriPattern, currentlyOpen };
  }

  static getDerivedStateFromProps(nextProps, prevState) {
    if (prevState.uriPattern.getOriginal() === nextProps.uriPattern) {
      return null;
    }

    return {
      uriPattern: new _uriPattern2.default(nextProps.uriPattern)
    };
  }

  componentDidMount() {
    for (const openItem of this.state.currentlyOpen) {
      this.registerCloseListener(openItem.stubItem, openItem);

      openItem.hydrateStub({
        copy: () => this.copyOpenItem(openItem)
      });
      if (this.props.className) {
        openItem.addClassName(this.props.className);
      }
    }

    this.subs.add(this.props.workspace.addOpener(this.opener));
  }

  render() {
    return this.state.currentlyOpen.map(item => {
      return _react2.default.createElement(
        _react.Fragment,
        { key: item.getKey() },
        item.renderPortal(this.props.children)
      );
    });
  }

  componentWillUnmount() {
    this.subs.dispose();
  }

  opener(uri) {
    const m = this.state.uriPattern.matches(uri);
    if (!m.ok()) {
      return undefined;
    }

    const openItem = new OpenItem(m);
    if (this.props.className) {
      openItem.addClassName(this.props.className);
    }

    return new Promise(resolve => {
      this.setState(prevState => ({
        currentlyOpen: [...prevState.currentlyOpen, openItem]
      }), () => {
        const paneItem = openItem.create({
          copy: () => this.copyOpenItem(openItem)
        });
        this.registerCloseListener(paneItem, openItem);
        resolve(paneItem);
      });
    });
  }

  copyOpenItem(openItem) {
    const m = this.state.uriPattern.matches(openItem.getURI());
    if (!m.ok()) {
      return null;
    }

    const stub = _stubItem2.default.create('generic', openItem.getStubProps(), openItem.getURI());

    const copiedItem = new OpenItem(m, stub.getElement(), stub);
    this.setState(prevState => ({
      currentlyOpen: [...prevState.currentlyOpen, copiedItem]
    }), () => {
      this.registerCloseListener(stub, copiedItem);
      copiedItem.hydrateStub({
        copy: () => this.copyOpenItem(copiedItem)
      });
    });

    return stub;
  }

  registerCloseListener(paneItem, openItem) {
    const sub = this.props.workspace.onDidDestroyPaneItem(({ item }) => {
      if (item === paneItem) {
        sub.dispose();
        this.subs.remove(sub);
        this.setState(prevState => ({
          currentlyOpen: prevState.currentlyOpen.filter(each => each !== openItem)
        }));
      }
    });

    this.subs.add(sub);
  }
}

exports.default = PaneItem; /**
                             * A subtree rendered through a portal onto a detached DOM node for use as the root as a PaneItem.
                             */

PaneItem.propTypes = {
  workspace: _propTypes2.default.object.isRequired,
  children: _propTypes2.default.func.isRequired,
  uriPattern: _propTypes2.default.string.isRequired,
  className: _propTypes2.default.string
};
class OpenItem {

  constructor(match, element = null, stub = null) {
    this.id = this.constructor.nextID;
    this.constructor.nextID++;

    this.domNode = element || document.createElement('div');
    this.domNode.tabIndex = '-1';
    this.domNode.onfocus = this.onFocus.bind(this);
    this.stubItem = stub;
    this.match = match;
    this.itemHolder = new _refHolder2.default();
  }

  getURI() {
    return this.match.getURI();
  }

  create(extra = {}) {
    const h = this.itemHolder.isEmpty() ? null : this.itemHolder;
    return (0, _helpers.createItem)(this.domNode, h, this.match.getURI(), extra);
  }

  hydrateStub(extra = {}) {
    if (this.stubItem) {
      this.stubItem.setRealItem(this.create(extra));
      this.stubItem = null;
    }
  }

  addClassName(className) {
    this.domNode.classList.add(className);
  }

  getKey() {
    return this.id;
  }

  getStubProps() {
    if (!this.itemHolder.isEmpty()) {
      const item = this.itemHolder.get();
      return {
        title: item.getTitle ? item.getTitle() : null,
        iconName: item.getIconName ? item.getIconName() : null
      };
    } else {
      return {};
    }
  }

  onFocus() {
    return this.itemHolder.map(item => item.focus && item.focus());
  }

  renderPortal(renderProp) {
    return _reactDom2.default.createPortal(renderProp({
      itemHolder: this.itemHolder,
      params: this.match.getParams(),
      uri: this.match.getURI()
    }), this.domNode);
  }
}
OpenItem.nextID = 0;
//# sourceMappingURL=data:application/json;charset=utf-8;base64,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