'use strict';

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.getNullActionPipelineManager = getNullActionPipelineManager;
function partial(fn, ...args) {
  return function wrapped() {
    return fn(...args);
  };
}

class ActionPipeline {
  constructor(actionKey) {
    this.actionKey = actionKey;
    this.middleware = [];
  }

  run(fn, repository, ...args) {
    const pipelineFn = this.middleware.map(middleware => middleware.fn).reduceRight((acc, nextFn) => partial(nextFn, acc, repository, ...args), partial(fn, ...args));
    return pipelineFn(...args);
  }

  addMiddleware(name, fn) {
    if (!name) {
      throw new Error('Middleware must be registered with a unique middleware name');
    }

    if (this.middleware.find(middleware => middleware.name === name)) {
      throw new Error(`A middleware with the name ${name} is already registered`);
    }

    this.middleware.push({ name, fn });
  }
}

exports.ActionPipeline = ActionPipeline;
class ActionPipelineManager {
  constructor({ actionNames } = { actionNames: [] }) {
    this.pipelines = new Map();
    this.actionKeys = {};
    actionNames.forEach(actionName => {
      const actionKey = Symbol(actionName);
      this.actionKeys[actionName] = actionKey;
      this.pipelines.set(actionKey, new ActionPipeline(actionKey));
    });
  }

  getPipeline(actionKey) {
    const pipeline = this.pipelines.get(actionKey);
    if (actionKey && pipeline) {
      return pipeline;
    } else {
      throw new Error(`${actionKey} is not a known action`);
    }
  }
}

exports.default = ActionPipelineManager;
class NullActionPipeline extends ActionPipeline {
  run(fn, repository, ...args) {
    return fn(...args);
  }

  addMiddleware() {
    throw new Error('Cannot add middleware to a null pipeline');
  }
}

const nullPipeline = new NullActionPipeline();

class NullActionPipelineManager extends ActionPipelineManager {
  getPipeline() {
    return nullPipeline;
  }
}

const nullActionPipelineManager = new NullActionPipelineManager();
function getNullActionPipelineManager() {
  return nullActionPipelineManager;
}
//# sourceMappingURL=data:application/json;charset=utf-8;base64,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