(function() {
  var CharacterPattern, _;

  _ = require('underscore-plus');

  CharacterPattern = /[^\s]/;

  module.exports = {
    activate: function() {
      return this.commandDisposable = atom.commands.add('atom-text-editor', {
        'autoflow:reflow-selection': (function(_this) {
          return function(event) {
            return _this.reflowSelection(event.currentTarget.getModel());
          };
        })(this)
      });
    },
    deactivate: function() {
      var ref;
      if ((ref = this.commandDisposable) != null) {
        ref.dispose();
      }
      return this.commandDisposable = null;
    },
    reflowSelection: function(editor) {
      var range, reflowOptions, reflowedText;
      range = editor.getSelectedBufferRange();
      if (range.isEmpty()) {
        range = editor.getCurrentParagraphBufferRange();
      }
      if (range == null) {
        return;
      }
      reflowOptions = {
        wrapColumn: this.getPreferredLineLength(editor),
        tabLength: this.getTabLength(editor)
      };
      reflowedText = this.reflow(editor.getTextInRange(range), reflowOptions);
      return editor.getBuffer().setTextInRange(range, reflowedText);
    },
    reflow: function(text, arg) {
      var block, blockLines, currentLine, currentLineLength, escapedLinePrefix, firstLine, i, j, leadingVerticalSpace, len, len1, linePrefix, linePrefixTabExpanded, lines, paragraphBlocks, paragraphs, ref, segment, tabLength, tabLengthInSpaces, trailingVerticalSpace, wrapColumn, wrappedLinePrefix;
      wrapColumn = arg.wrapColumn, tabLength = arg.tabLength;
      paragraphs = [];
      text = text.replace(/\r\n?/g, '\n');
      leadingVerticalSpace = text.match(/^\s*\n/);
      if (leadingVerticalSpace) {
        text = text.substr(leadingVerticalSpace.length);
      } else {
        leadingVerticalSpace = '';
      }
      trailingVerticalSpace = text.match(/\n\s*$/);
      if (trailingVerticalSpace) {
        text = text.substr(0, text.length - trailingVerticalSpace.length);
      } else {
        trailingVerticalSpace = '';
      }
      paragraphBlocks = text.split(/\n\s*\n/g);
      if (tabLength) {
        tabLengthInSpaces = Array(tabLength + 1).join(' ');
      } else {
        tabLengthInSpaces = '';
      }
      for (i = 0, len = paragraphBlocks.length; i < len; i++) {
        block = paragraphBlocks[i];
        linePrefix = block.match(/^\s*(\/\/|\/\*|;;|#'|\|\|\||--|[#%*>-])?\s*/g)[0];
        linePrefixTabExpanded = linePrefix;
        if (tabLengthInSpaces) {
          linePrefixTabExpanded = linePrefix.replace(/\t/g, tabLengthInSpaces);
        }
        blockLines = block.split('\n');
        if (linePrefix) {
          escapedLinePrefix = _.escapeRegExp(linePrefix);
          blockLines = blockLines.map(function(blockLine) {
            return blockLine.replace(RegExp("^" + escapedLinePrefix), '');
          });
        }
        blockLines = blockLines.map(function(blockLine) {
          return blockLine.replace(/^\s+/, '');
        });
        lines = [];
        currentLine = [];
        currentLineLength = linePrefixTabExpanded.length;
        wrappedLinePrefix = linePrefix.replace(/^(\s*)\/\*/, '$1  ').replace(/^(\s*)-(?!-)/, '$1 ');
        firstLine = true;
        ref = this.segmentText(blockLines.join(' '));
        for (j = 0, len1 = ref.length; j < len1; j++) {
          segment = ref[j];
          if (this.wrapSegment(segment, currentLineLength, wrapColumn)) {
            if (firstLine !== true) {
              if (linePrefix.search(/^\s*\/\*/) !== -1 || linePrefix.search(/^\s*-(?!-)/) !== -1) {
                linePrefix = wrappedLinePrefix;
              }
            }
            lines.push(linePrefix + currentLine.join(''));
            currentLine = [];
            currentLineLength = linePrefixTabExpanded.length;
            firstLine = false;
          }
          currentLine.push(segment);
          currentLineLength += segment.length;
        }
        lines.push(linePrefix + currentLine.join(''));
        paragraphs.push(lines.join('\n').replace(/\s+\n/g, '\n'));
      }
      return leadingVerticalSpace + paragraphs.join('\n\n') + trailingVerticalSpace;
    },
    getTabLength: function(editor) {
      var ref;
      return (ref = atom.config.get('editor.tabLength', {
        scope: editor.getRootScopeDescriptor()
      })) != null ? ref : 2;
    },
    getPreferredLineLength: function(editor) {
      return atom.config.get('editor.preferredLineLength', {
        scope: editor.getRootScopeDescriptor()
      });
    },
    wrapSegment: function(segment, currentLineLength, wrapColumn) {
      return CharacterPattern.test(segment) && (currentLineLength + segment.length > wrapColumn) && (currentLineLength > 0 || segment.length < wrapColumn);
    },
    segmentText: function(text) {
      var match, re, segments;
      segments = [];
      re = /[\s]+|[^\s]+/g;
      while (match = re.exec(text)) {
        segments.push(match[0]);
      }
      return segments;
    }
  };

}).call(this);

//# sourceMappingURL=data:application/json;base64,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
